﻿#pragma TextEncoding	= "UTF-8"
#pragma rtGlobals 		= 3
#pragma IgorVersion		= 6.36
#pragma ModuleName		= SpectraGraphTools
#pragma version			= 1.4
#pragma DefaultTab		= {3,20,4}																// Set default tab width in Igor Pro 9 and later

// --------------------- Project Updater header ----------------------
// If you're using Igor Pro 8 or later and have Project Updater
// installed, this package can check periodically for new releases.
// https://www.wavemetrics.com/project/Updater
static Constant kProjectID = 21562																// the project node on IgorExchange
static StrConstant ksShortTitle = "Spectra Graph Tools"											// the project short title on IgorExchange

//________________________________________________________________________________________________
//	Written by Stephan Thuermer - https://www.wavemetrics.com/user/chozo
//	Original code & idea for the right-click graph menus by Tony Withers - https://www.wavemetrics.com/user/tony
//
//	This is a collection of functions for modifying traces and graphs.
//	Contents:
//		- Basic Graph Style and Label Macros
//		- Quick Normalization :		Normalize to unit area or height / scale the traces in a graph
//		- SpectraWaterfall :		Generates a waterfall plot out of 2D data or a folder
//		- BuildTraceOffsetPanel :	A small panel to offset and color traces
//		- BuildGraphSizePanel :		A small panel to set the graph size and margins
//
// 2021-03-01 - ver. 1.00:	Initial public release.
// 2021-03-07 - ver. 1.10:	Quick normalization did not work properly when the data included NaNs.
//							Graph Size panel: Axes margins did not copy correctly from one graph to the other.
//							The Graph Size panel threw an error on Mac because of a wrong ModifyControl command.
//							Trace Offset panel: Added keyboard shortcuts for selecting and moving the cursor.
//							Trace Offset panel: It is now possible to move and scale selected traces with the cursor keys.
// 2021-03-11 - ver. 1.11:	A cursor set on an image triggered an error message for the TraceOffset panel.
// 2021-03-16 - ver. 1.20:	Added global constant for the multiplier step and made keyboard control more precise.
//							Added Offset Traces functionality to the graph and traces right-click menus.
// 2021-03-17 - ver. 1.21:	Now it is possible to choose whether offsets are set or added.
//							Fixed bug: Closing a graph did not properly update active window.
//							Now the total shift is also displayed for single traces.
// 2021-03-28 - ver. 1.30:	Graph Size panel: The last selected size unit setting is now saved in the current experiment.
//							Trace offset panel: Fixed a few error messages triggered by very old versions of the panel.
//							Trace offset panel: Trace scaling increment steps now get smaller and bigger depending on the
//							order of magnitude of the current scaling (helps to work with very big or small scaling factors).
//							Trace offset panel: Added a button to quickly set a new cursor onto the graph.
//							The currently selected trace name is now displayed in the panel title.
// 2021-05-14 - ver. 1.40:	Slightly adjusted the axis-label graph macros.
//							Graph size panel: Now switching the axes keeps the axis label visible.
//							Offset menu: Works now for 1-column 2D waves as well.
//							Graph Size panel: Hold shift while changing the font size to change the global axes font instead.
////						Graph Size panel: Added print checkbox to enable print-to-history function for most controls.
//________________________________________________________________________________________________

Static Constant kOffsetIncrStep = 0.005															// fraction of the total range to increase / decrease values each step
Static Constant kMultiIncrStep = 0.1															// by how much the multiplier is incremented (keyboard presses are 1/5 of this value)
Static StrConstant ksTraceIgnoreList = "w_base;MQP_*;"											// list of traces to ignore, wildcards okay
Static StrConstant ksGraphSizeUnitSave = "V_SizePanelUnit"										// global variable name which saves the current unit setting of the Graph Size Panel

//-------------------------------- functions for user code ---------------------------------------
Static Function TraceTool_FetchXShift(input, unit)												// extract predetermined X offset shift values from data
	Wave input
	String &unit																				// optional unit string
	Variable xShift = NaN																		// the shift value which will be preset in the panel
	// ++++++++++++++++++++++ get the photon energy as shift value +++++++++++++++++++++++++++++++
	Variable neg = 1
	String notes = note(input)
	String energy = StringByKey("Xray energy", notes, "=","\r")									// find the photon energy
	if (!strlen(energy))
		energy = StringByKey("photon energy ", notes, "=","\r")									// try a different key
	endif
	if (!strlen(energy))
		energy = StringByKey("Excitation Energy", notes, "=","\r")								// try yet a different key
		neg = -1																				// this data requires flipping
	endif
	unit = "eV"
	xShift = str2num(energy)*neg
	// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	return xShift 																				// return NaN for no change and return negative values for flipping the trace
End

Static Function TraceTool_FetchYShift(input,unit)												// extract predetermined Y offset shift values
	Wave input
	String &unit
	return NaN	 																				// return NaN for no change and return negative values for flipping the trace
End
//------------------------------------------------------------------------------------------------

Menu "Spectra Tools"
	"-"
	"Normalize Traces to 1",/Q, ScaleTo1()
	"Normalize Traces to 1 at Cursor A",/Q, ScaleTo1(xpos=hcsr(A))
	"Normalize Traces to Area",/Q, ScaleTo1(toarea=1)
	"-"
	"Waterfall Plot of 2D Data or Folder", /Q, CreateBrowser; SpectraWaterfall()
	"Quick Colorize Traces", /Q, QuickColorAllTraces()
	"Graph Size Panel ...", /Q, BuildGraphSizePanel()
	"Trace Offset Panel ...", /Q, BuildTraceOffsetPanel()
End

Menu "GraphPopup"
	"Left: Autoscale Only Visible", /Q,	SetAxis/A=2/E=0 left
	"Left: Autoscale From Zero", /Q, SetAxis/A=2/E=1 left
	"Traces: Undo All Scaling", /Q, ModifyGraph muloffset={0,0}
	"Traces: Undo All Offsets", /Q, ModifyGraph offset={0,0}
	Submenu "Offset Spectra"
		"Spread Traces Across y Range",	/Q, SpectraGraphTools#SpreadToFillRange()
		"Spread Traces: Equal Offset", 	/Q, SpectraGraphTools#SpreadSpectra(0)
		"Spread Traces: Equal Gap", 	/Q, SpectraGraphTools#SpreadSpectra(1)
	End
End

Menu "TracePopup", dynamic
	"Left: Autoscale Only Visible", /Q,	SetAxis/A=2/E=0 left
	"Left: Autoscale From Zero", /Q, SetAxis/A=2/E=1 left
	"Traces: Undo All Scaling", /Q, ModifyGraph muloffset={0,0}
	"Traces: Undo All Offsets", /Q, ModifyGraph offset={0,0}
	Submenu "Offset Spectra"
		SpectraGraphTools#SaveMenuPosition("y-Offset: Align Traces Here"),/Q, SpectraGraphTools#AlignTracesAtXPos(0)	// holding ctrl / command aligns to zero
		SpectraGraphTools#SaveMenuPosition("y-Scale: Match Traces Here"), /Q, SpectraGraphTools#AlignTracesAtXPos(1)	// holding ctrl / command scales to one
		SpectraGraphTools#SaveMenuPosition("y-Order Traces Here"),		/Q, SpectraGraphTools#OrderTracesAtXPos()		// holding ctrl / command sorts in descending order
		"Spread Traces Across y Range", /Q, SpectraGraphTools#SpreadToFillRange()
		"Spread Traces: Equal Offset", 	/Q, SpectraGraphTools#SpreadSpectra(0)
		"Spread Traces: Equal Gap", 	/Q, SpectraGraphTools#SpreadSpectra(1)
	End
End

//________________________________________________________________________________________________
//
//										Standard Graph Macros
//________________________________________________________________________________________________

Proc BoxStyle() : GraphStyle
	PauseUpdate; Silent 1
	GetAxis/Q right
	if (V_flag == 1)
		ModifyGraph/Z mirror(left)=2
	endif
	GetAxis/Q top
	if (V_flag == 1)
		ModifyGraph/Z mirror(bottom)=2
	endif
	ModifyGraph noLabel=0
	ModifyGraph standoff=0
	ModifyGraph ZisZ=1
	ModifyGraph tick=0
	ModifyGraph notation(left)=1
	ModifyGraph axisOnTop=1
	ModifyGraph minor=1
	if ((GetKeyState(0) & 2^0) != 0)															// ctrl / command = paper style (no left axis notation)
		ModifyGraph tick(left)=3
		ModifyGraph noLabel(left)=1
	endif
	ResumeUpdate
EndMacro

Proc XAS_Label() : GraphStyle
	Label/Z left "\u#2Intensity (arb. u.) \\E"
	Label/Z bottom "Photon energy (\U)"
EndMacro

Proc XES_Label() : GraphStyle
	Label/Z left "\u#2Intensity (arb. u.) \\E"
	Label/Z bottom "Emission energy (\U)"
EndMacro

Proc TOF_Label() : GraphStyle
	Label/Z left "\u#2Intensity (arb. u.) \\E"
	Label/Z bottom "Flight time (\U)"
EndMacro

Proc PKE_Label() : GraphStyle
	Label/Z left "\u#2Intensity (arb. u.) \\E"
	Label/Z bottom "\u#2"+SelectString((GetKeyState(0) & 2^0) != 0, "Kinetic", "Binding")+" energy (eV)"	// BE label with ctrl / command
EndMacro

//________________________________________________________________________________________________
//	Change the Color of up to 13 Traces with Preset Colors - Sorted to give good contrast for even a few traces
//________________________________________________________________________________________________

Function QuickColorAllTraces()																	// colors traces with 12 different colors (hard-coded)
	String traces
	Variable trNum = FetchTraces("",traces,1)
	if (trNum == 0)
		return 0
	endif
	Make/FREE colors = {{65280,0,0}, {65280,43520,0}, {0,65280,0}, {0,52224,0}, {0,26214,1329}, {0,65280,65280}, {0,43520,65280}, {0,15872,65280}, {44253,29492,58982}, {65535,16385,55749}, {26411,1,52428}, {26112,26112,26112}, {0,0,0}}
	Variable trI, cI, cCount = DimSize(colors,1)
	Variable leap = trNum < cCount/2 ? trunc((cCount-1)/(trNum-1)) : 1							// introduce a leap value to choose more distant colors for only a few traces
	for (trI = 0, cI = 0; trI < trNum; trI += 1, cI += leap)
		cI = cI == cCount ? 0 : cI																// loop colors around
		ModifyGraph rgb($StringFromList(trI,traces))=(colors[0][cI],colors[1][cI],colors[2][cI])// set new color offset
	endfor
End

//________________________________________________________________________________________________
//
//							Scaling Normalization of Graph Traces
//________________________________________________________________________________________________

Function ScaleTo1([smth,xpos,toarea])
	Variable smth, xpos, toarea
	String traces  = TraceNameList("",";",1)
	Variable items = ItemsInList(Traces)
	if (items == 0)
		return 0
	endif

	Variable i, shiftBaseline = !(GetKeyState(0) == 4)											// don't shift background if shift is pressed
	for (i = 0; i < items; i += 1)
		String curr	= StringFromList(i,traces)
		Wave inwave	= TraceNameToWaveRef("", curr)
		if (DimSize(inwave,1) < 2 && WaveDims(inwave) < 3)
			Duplicate/FREE inwave work
		elseif (WaveDims(inwave) == 2)
			String range = StringByKey("YRANGE", TraceInfo("",curr,0))
			Variable col = strsearch(range,"[",Inf,1)
			range = range[col+1,inf]
			col = str2num(RemoveEnding(range,"]"))
			if (numtype(col) == 0)
				Duplicate/R=[][col]/FREE inwave work
			else
				continue
			endif
		else
			return -1
		endif
		MatrixOP/O work = replaceNaNs(work,0)
		
		if( !ParamIsDefault(smth) && smth > 0)													// optional smoothing before scaling
			Smooth smth, work
		endif
		
		Variable minval	= wavemin(work)*shiftBaseline
		Variable maxval	= wavemax(work)
		if (!ParamIsDefault(xpos) && numtype(xpos) == 0)
			Variable pntpos = x2pnt(work, xpos)
			if (pntpos > 0 && pntpos < DimSize(work,0)-1)										// make sure to stay within range
				maxval = work(xpos)
			endif
		endif
		Variable scale = maxval-minval
		if (!ParamIsDefault(toarea) && toarea == 1)
			scale  = area(work)
			minval = 0
		endif
		scale = scale == 0 ? 1 : scale
		ModifyGraph offset($curr)={,-minval/scale},muloffset($curr)={,1/scale}
	endfor
	return 0
End

//________________________________________________________________________________________________
//
//					Creates Waterfall Plot from a 2D Wave or a Folder of Waves
//________________________________________________________________________________________________

Function SpectraWaterfall()
	Wave/Z inwave = $GetBrowserSelection(0)														// get the first selected wave
	if (!WaveExists(inwave))
		Print "Nothing selected"
		return -1
	endif
	String list	= ""
	Variable OneD = WaveDims(inwave) == 1														// first check, if the incoming wave is 1D or 2D
	DFREF saveDFR = GetDataFolderDFR()
	if (OneD)
		SetDataFolder GetWavesDataFolderDFR(inwave)
		list = WaveList("!Int_*", ";", "DIMS:1")
		list = SortList(list, ";", 16)
	endif
	
	Display/W=(230,50,650,450)/K=1/N=$UniqueName("WaterfallGraph", 6, 0) as "Waterfall of "+NameOfWave(inwave)
	Variable i, items = OneD ? ItemsInList(list) : DimSize(inwave,1)
	for (i = items-1; i > -1; i -=1)															// go backwards (in reverse order, so that the later traces are BEHIND the earlier ones)
		if (OneD)
			AppendToGraph/Q $Stringfromlist(i,list)
		else
			AppendToGraph/Q inwave[][i]
		endif
	endfor
	SetDataFolder saveDFR
	
	Execute/Q "BoxStyle()"
	ColorSpectrumFromTable(0, "Rainbow")
	
	list = TraceNameList("",";",1)
	Variable initOffset	= WaveMax(inwave) * 0.05												// the initial distance will be set in percent from the maximal value of the selected wave
	for (i = 1; i < items; i += 1)
		ModifyGraph offset($StringFromList(items-1 - i,list))={,i*initOffset}
	endfor
	return 0
End

//________________________________________________________________________________________________
//
//							Right-Click Offset Menus for Graph Traces
//				Adapted from the StackTraces procedures v1.20 by Tony Withers
//________________________________________________________________________________________________

Static Function/S SaveMenuPosition(s_menu)														// use to create a menu item in a popup when you need to know where the user clicks to invoke the popup
	String s_menu
	if (WinType("") != 1)																		// don't create variables if Igor is just rebuilding the menu
		return ""
	endif
	GetMouse/W=kwTopWin
	if (V_left<0 || V_top<0)
		return ""
	endif
	if(IgorVersion() < 8.00)																	// compatibility with earlier Igor versions: save mouse position as global variable
		Variable/G V_menuX = V_left, V_menuY = V_top
	endif
	return s_menu
End

//################################################################################################

Static Function getYfromWave(gName, trace, xPos, yVal, yMin)
	String gName, trace
	Variable xPos, &yVal, &yMin
	
	Wave w = TraceNameToWaveRef(gName, trace)
	Wave/Z w_x = XWaveRefFromTrace(gName, trace)
	
	String range
	Variable col2D = 0
	if (WaveDims(w) == 2 && DimSize(w,1) > 1)													// handle 2D waterfall plots => extract column
		range = StringByKey("YRANGE", TraceInfo(gName,trace,0))									// extract the last number out of something like "[*][15]" or "[3,50][15]"
		col2D = strsearch(range,"[",Inf,1)
		range = range[col2D+1,inf]
		col2D = str2num(RemoveEnding(range,"]"))
		if (numtype(col2D) == 0)
			Duplicate/R=[][col2D]/FREE w work
			Wave w = work
		else
			return -1
		endif
	endif
	
	yMin = WaveMin(w)
	yVal = NaN
	if(WaveExists(w_x))
		FindLevel/P/Q w_x, xPos
		if (!v_flag && v_levelX > 0 && v_levelX < DimSize(w,0)-1)								// make sure to stay within range
			yVal = w[round(v_levelX)]
		else
			return -1
		endif
	else
		if (x2pnt(w, xPos) > 0 && x2pnt(w, xPos) < DimSize(w,0)-1)
			yVal = w(xPos)
		else
			return -1
		endif
	endif
	return 0
End

//################################################################################################

Static Function getMinMaxfromWave(gName, trace, yMin, yMax)										// figure out plotted x-range for trace and finds wave's max and min values in this range
	String gName, trace
	Variable &yMin, &yMax 
	
	wave w = TraceNameToWaveRef(gName, trace)
	wave/Z w_x = XWaveRefFromTrace(gName, trace)
	
	String range
	Variable col2D = 0
	if (WaveDims(w) == 2 && DimSize(w,1) > 1)
		range = StringByKey("YRANGE", TraceInfo(gName,trace,0))
		col2D = strsearch(range,"[",Inf,1)
		range = range[col2D+1,inf]
		col2D = str2num(RemoveEnding(range,"]"))
		if (numtype(col2D) == 0)
			Duplicate/R=[][col2D]/FREE w work
			Wave w = work
		else
			return -1
		endif
	endif
	
	Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0											// x-offset and y-scale aware
	getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
	if (WhichListItem("bottom",AxisList(gName)) > -1)
		GetAxis/W=$gName/Q bottom
	elseif (WhichListItem("top",AxisList(gName)) > -1)
		GetAxis/W=$gName/Q top
	else
		return -1
	endif
	
	if(WaveExists(w_x))
		FindLevel /Q w_x, v_min																	// no /P flag here - assumes same x-scaling for w and w_x
		v_min = v_flag ? 0 : V_LevelX
		FindLevel /Q w_x, v_max
		v_max = v_flag ? numpnts(w) : V_LevelX
	endif
	Variable fr = (v_min - xOff)/xScale
	Variable to = (v_max - xOff)/xScale
	yMax = WaveMax(w, fr, to)*yScale
	yMin = WaveMin(w, fr, to)*yScale
	return 0
End

//################################################################################################

Static Function AlignTracesAtXPos(which)														// use trace popup to y-align traces at clicked x position by setting offsets (which = 0) or multiplicators (which = 1)
	Variable which
	
	GetLastUserMenuInfo																			// figure out graph and trace names
#if(IgorVersion() < 8.00)																		// compatibility with earlier Igor versions
	NVAR V_mouseX = v_menuX
	NVAR V_mouseY = v_menuY
#endif

	String gName = S_graphName, trace = S_traceName
	String s_info = TraceInfo(gName, trace, 0), traces = ""
	Variable trNum = FetchTraces("",traces,1)
	Variable xClick = AxisValFromPixel(gName, StringByKey("XAXIS",s_info) , V_mouseX)
	Variable yClick = AxisValFromPixel(gName, StringByKey("YAXIS",s_info) , V_mouseY)
	
	Variable normVal = (GetKeyState(0) & 2^0)													// holding ctrl / command aligns to zero or scales to one
	yClick = normVal && which == 0 ? 0 : yClick													// shift to zero?
	
	Variable i, err, modVal
	for(i = 0; i < trNum; i += 1)
		trace = StringFromList(i,traces)
		Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0, ywVal, ywMin
		getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
		err = getYfromWave(gName, trace, (xClick-xOff)/xScale, ywVal, ywMin)					// extract wave min and Y value at X position
		if (err)
			continue
		endif
		if (which)
			modVal = normVal ? 1/(ywVal-ywMin) : (yClick-yOff/yScale-ywMin)/(ywVal-ywMin)		// scale to one?
			ModifyGraph muloffset($trace)={,modVal}												// scale to match position
			if (normVal && which)																// if normalized scale offsets down as well (because the scale difference can be huge)
				ModifyGraph offset($trace)={,-ywMin*modVal}
			endif
		else
			modVal = yClick-ywVal*yScale
			ModifyGraph offset($trace)={,modVal}												// shift to align position
		endif
	endfor
	return 0
End

//################################################################################################

Static Function OrderTracesAtXPos()																// reorder traces based on (possibly offset) value at x=xpos
	GetLastUserMenuInfo																			// figure out graph and trace names
#if(IgorVersion() < 8.00)																		// compatibility with earlier Igor versions
	NVAR V_mouseX = v_menuX
#endif

	String gName = S_graphName, trace = S_traceName
	String s_info = TraceInfo(gName, trace, 0), traces = ""
	Variable trNum = FetchTraces("",traces,1)
	Variable xClick = AxisValFromPixel(gName, StringByKey("XAXIS",s_info) , V_mouseX)
	
	Make/Free/T/N=(trNum) w_traces
	Make/Free/N=(trNum) w_order
	
	Variable i, err
	for(i = 0 ; i < trNum; i += 1)
		trace = StringFromList(i,traces)
		Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0, ywVal, ywMin
		getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
		err = getYfromWave(gName, trace, (xClick-xOff)/xScale, ywVal, ywMin)					// extract wave min and Y value at X position
		w_traces[i] = trace
		w_order[i] = err ? yOff : yOff+ywVal*yScale
	endfor
	
	if (GetKeyState(0) & 2^0)																	// holding ctrl / command triggers descending order sorting
		Sort/R w_order, w_order, w_traces
	else
		Sort w_order, w_order, w_traces
	endif
	
	for(i = 0; i < numpnts(w_traces); i += 1)
	#if(IgorVersion() > 7.00)
		ReorderTraces _front_, {$w_traces[i]}
	#endif
	endfor
	return 0
End

//################################################################################################

Static Function SpreadToFillRange()																// spread (equally offset) spectra over Y axis range; traces are stacked in the order of plotting
	String traces = ""
	Variable trNum = FetchTraces("",traces,1)
	Make/Free/N=(trNum)/T w_traces = StringFromList(p,traces)
	Make/Free/N=(trNum) w_offsets = getTraceOffset("", w_traces[p], 1)

	Variable yMin, yMax, low, high
	getMinMaxfromWave("", w_traces[0], yMin, yMax);			low  = yMin + w_offsets[0]			// find low from first trace
	getMinMaxfromWave("", w_traces[trNum-1], yMin, yMax);	high = yMax + w_offsets[trNum-1]	// find high from last trace
	
	if (WhichListItem("left",AxisList("")) > -1)
		GetAxis/Q left
	elseif (WhichListItem("right",AxisList("")) > -1)
		GetAxis/Q right
	else
		return -1
	endif
	
	Variable bottomGap = low-v_Min, topGap = v_Max-high
	Variable i, dOffset = (topGap+bottomGap)/(trNum-1)
	for (i = 0; i < trNum; i += 1)																// apply offsets
		ModifyGraph offset($w_traces[i])={,w_offsets[i]- bottomGap + dOffset*i}
	endfor
	return 0
End

//################################################################################################

Static Function SpreadSpectra(which)
	Variable which																				// which: 0 = equal offset increment, 1 = equal spacing
	
	String traces = ""
	Variable trNum = FetchTraces("",traces,1)
	Make/Free/N=(trNum)/T w_traces = StringFromList(p,traces)
	Make/Free/N=(trNum)/Wave w_waves = TraceNameToWaveRef("",w_traces)
	Make/Free/N=(trNum) w_low, w_high, w_range	
	
	Variable i, yMin, yMax
	for (i = 0; i < trNum; i += 1)
		getMinMaxfromWave("", w_traces[i], yMin, yMax);  w_low[i] = yMin;  w_high[i] = yMax;
	endfor
	w_range = w_high-w_low
	
	if (WhichListItem("left",AxisList("")) > -1)
		GetAxis/Q left
	elseif (WhichListItem("right",AxisList("")) > -1)
		GetAxis/Q right
	else
		return -1
	endif
	
	Variable yAxMin = v_min, yAxRange = (v_max-v_min), dataRange = sum(w_range)
	Variable gap = which ? (yAxRange-dataRange)/(trNum-1) : (yAxRange-w_range[trNum-1])/(trNum-1)	// gap or offset between each trace (defined as distance between highest value of spectrum n and lowest value of spectrum n+1)
	Variable newOff
	for (i = 0; i < trNum; i += 1)
		if (i == 0)
			newOff = yAxMin - w_low[0]															// add this to bring first trace down to bottom
		else
			newOff = which ? w_high[i-1] - w_low[i] + gap : w_low[i-1] - w_low[i] + gap
		endif
		ModifyGraph offset($w_traces[i])={,newOff}
		w_high[i] += newOff
		w_low[i]  += newOff
	endfor
	return 0
End

//________________________________________________________________________________________________
//
//										TRACE OFFSET PANEL
//________________________________________________________________________________________________

Function BuildTraceOffsetPanel()
	if (WinType("OffsetPanel"))
		DoWindow/F OffsetPanel
		return 0
	endif
	NewPanel/K=1/W=(450,60,885,475)/N=OffsetPanel as "Modify Traces"
	ModifyPanel /W=OffsetPanel ,fixedSize= 1 ,noEdit=1
	SetWindow OffsetPanel hook(UpdateHook)=OffsetPanelUpdateFunc

	SetDrawLayer UserBack
	DrawLine 15,280,205,280
	DrawLine 15,345,205,345
	DrawLine 230,235,420,235
	DrawLine 230,317,420,317
	// +++++++++++++++++++++++++++++++++++++++ left side +++++++++++++++++++++++++++++++++++++++++
	GroupBox MultiTraceBox		,pos={6,7}		,size={208,372}	,title="Multiple Traces"		,fstyle=1		,fColor=(0,16000,65535)
	CheckBox cHiddenTraces		,pos={28,28}	,size={60,22}	,title="Include Hidden Traces"	,value = 1		,help={"Include hidden traces when scaling."}
	SetVariable vaSetYOffset	,pos={15,70}	,size={90,16}	,title="y:"
	SetVariable vaScaleYOffset	,pos={15,110}	,size={90,16}	,title="y:"
	SetVariable vaShiftYOffset	,pos={15,150}	,size={90,16}	,title="y:"
	SetVariable vaSetXOffset	,pos={115,70}	,size={90,16}	,title="x:"
	SetVariable vaScaleXOffset	,pos={115,110}	,size={90,16}	,title="x:"
	SetVariable vaShiftXOffset	,pos={115,150}	,size={90,16}	,title="x:"
	
	CheckBox ChooseSet			,pos={15,50}	,size={60,22}	,title="Set or "				,value=1		,help={"Sets and overrides offsets of traces."}
	CheckBox ChooseAdd			,pos={68,50}	,size={60,22}	,title="Add"					,value=0		,help={"Adds to the current trace offsets."}
	TitleBox SetTitle			,pos={111,50}	,size={160,13}	,title="Offset per Trace"
	TitleBox ScaleTitle			,pos={15,93}	,size={160,13}	,title="Scale Offset per Trace"
	TitleBox ShiftTitle			,pos={15,133}	,size={160,13}	,title="Shift Offset of All Traces"
	Button ResetYButton			,pos={15,175}	,size={90,22}	,title="Y Reset"								,help={"Resets all Y offsets to 0."}
	Button ResetXButton			,pos={115,175}	,size={90,22}	,title="X Reset"								,help={"Resets all X offsets to 0."}
	Button ReverseButton		,pos={15,205}	,size={190,22}	,title="Reverse Sorting"						,help={"Reverses the order in which the traces are sorted in the graph."}
	SetVariable vTraceFilter	,pos={15,235}	,size={190,16}	,title="Filter:"								,help={"Apply offsets only to traces with matching name. You can use wildcard characters."}
	CheckBox cOmitFilter		,pos={48,257}	,size={90,22}	,title="Filter Excludes Traces"	,value = 0		,help={"The filter keyword is used to omit traces instead of including them."}

	PopupMenu ColorSelect		,pos={15,290}	,size={190,22}									,bodyWidth=190
	Button ColorButton			,pos={15,315}	,size={110,22}	,title="Colorize"								,help={"Colorize traces with currently selected color spectrum. To reverse spectrum hold 'ctrl'/'command'."}
	CheckBox cReverseColor		,pos={135,318}	,size={60,22}	,title="in Reverse"				,value = 0		,help={"Applies the color in reverse (also works directly by holding 'ctrl'/'command' while pressing the button.)"}
	PopUpMenu ProfileSelect		,pos={15,352}	,size={190,20}	,title="Profile"				,bodywidth=152
	// +++++++++++++++++++++++++++++++++++++++ right side ++++++++++++++++++++++++++++++++++++++++
	GroupBox SingleTraceBox		,pos={220,7}	,size={208,372}	,title="Individual Traces"		,fstyle=1		,fColor=(0,16000,65535)
	Button CsrActButton			,pos={225,25}	,size={60,22}	,title="Cursor:"				,fstyle=1		,help={"Sets a new cursor on the graph."}
	CheckBox traceCsrA			,pos={290,28}	,size={60,22}	,title="A"										,help={"Selects cursor A (a key)."}
	CheckBox traceCsrB			,pos={325,28}	,size={60,22}	,title="B"										,help={"Selects cursor B (b key)."}
	CheckBox traceCsrC			,pos={360,28}	,size={60,22}	,title="C"										,help={"Selects cursor C (c key)."}
	CheckBox traceCsrD			,pos={395,28}	,size={60,22}	,title="D"										,help={"Selects cursor D (d key)."}
	
	TitleBox trSetTitle			,pos={230,50}	,size={170,13}	,title="Set Trace Offset"
	TitleBox trScaleTitle		,pos={230,93}	,size={170,13}	,title="Set Trace Multiplier"
	TitleBox trShiftTitle		,pos={230,133}	,size={170,13}	,title="Reposition at Cursor"
	SetVariable trSetYOffset	,pos={230,70}	,size={90,16}	,title="y:"
	SetVariable trScaleYOffset	,pos={230,110}	,size={90,16}	,title="y:"
	SetVariable trShiftYOffset	,pos={230,150}	,size={90,16}	,title="y:"
	SetVariable trSetXOffset	,pos={330,70}	,size={90,16}	,title="x:"
	SetVariable trScaleXOffset	,pos={330,110}	,size={90,16}	,title="x:"
	SetVariable trShiftXOffset	,pos={330,150}	,size={90,16}	,title="x:"
	SetVariable trScaleNorm		,pos={230,178}	,size={140,16}	,title="Normalize:"				,bodyWidth=80	,help={"Adjusts the multiplier to normalize the trace height to this value at the current cursor's position (excluding the Y offset)."}
	Button trNormTo1Button		,pos={380,177}	,size={40,20}	,title="to 1"									,help={"Normalizes trace height to 1 at current cursor position (excluding the Y offset)."}
	Button trResetYButton		,pos={230,205}	,size={90,22}	,title="Y Reset"								,help={"Resets Y offset and multiplier to 0."}
	Button trResetXButton		,pos={330,205}	,size={90,22}	,title="X Reset"								,help={"Resets X offset and multiplier to 0."}
	
	TitleBox trCsrControlTitle	,pos={230,240}	,size={190,13}	,title="Control Cursor Position"
	Button trPrevTraceButton	,pos={230,258}	,size={90,22}	,title="< Prev Trace"							,help={"Moves the cursor to onto the previous trace in the list of traces (page down key)."}
	Button trNextTraceButton	,pos={330,258}	,size={90,22}	,title="Next Trace >"							,help={"Moves the cursor to onto the next trace in the list of traces (page up key)."}
	Button trGoToMaxButton		,pos={230,288}	,size={90,22}	,title="Go to Max"								,help={"Sets the cursor to trace's maximum (h key)."}
	Button trGoToMinButton		,pos={330,288}	,size={90,22}	,title="Go to Min"								,help={"Sets the cursor to trace's minimum (l key)."}
	
	SetVariable trOffxDelta		,pos={275,325}	,size={100,16}	,title="dx:"					,bodyWidth=80	,help={"Negative values will flip trace in X direction."}
	SetVariable trOffyDelta		,pos={275,353}	,size={100,16}	,title="dy:"					,bodyWidth=80	,help={"Negative values will flip trace in Y direction."}
	Button trAddXButton			,pos={385,323}	,size={35,20}	,title="+"			,fstyle=1	,fsize=14		,help={"Add X delta value."}
	Button trRemXButton			,pos={230,323}	,size={35,20}	,title="-"			,fstyle=1	,fsize=14		,help={"Subtract X delta value."}
	Button trAddYButton			,pos={385,351}	,size={35,20}	,title="+"			,fstyle=1	,fsize=14		,help={"Add Y delta value."}
	Button trRemYButton			,pos={230,351}	,size={35,20}	,title="-"			,fstyle=1	,fsize=14		,help={"Subtract Y delta value."}
	
	Button vaApplyAllButton		,pos={10,385}	,size={200,22}	,title="Apply All to Data (overwrite!)"			,help={"Applies all offsets and scales to 1D data. Will overwrite the original waves, so use this with care!"}
	Button trApplyCurButton		,pos={225,385}	,size={200,22}	,title="Apply to This Data (overwrite!)"		,help={"Applies offsets and scales to the currently selected 1D data. Will overwrite the original waves, so use this with care!"}
	// ++++++++++++++++++++++++++++++++++++++++ settings +++++++++++++++++++++++++++++++++++++++++
	SetVariable vaSetYOffset	,value=_NUM:0	,limits={0,inf,1}					,userdata="0"
	SetVariable vaScaleYOffset	,value=_NUM:1	,limits={0,inf,kMultiIncrStep}		,userdata="1"
	SetVariable vaShiftYOffset	,value=_NUM:0	,limits={-inf,inf,1}
	SetVariable vaSetXOffset	,value=_NUM:0	,limits={-inf,inf,0.1}				,userdata="0"
	SetVariable vaScaleXOffset	,value=_NUM:1	,limits={0,inf,kMultiIncrStep}		,userdata="1"
	SetVariable vaShiftXOffset	,value=_NUM:0	,limits={-inf,inf,1}
	
	SetVariable trSetYOffset	,value=_NUM:0	,limits={-inf,inf,1}
	SetVariable trScaleYOffset	,value=_NUM:1	,limits={-inf,inf,kMultiIncrStep}
	SetVariable trShiftYOffset	,value=_NUM:0	,limits={-inf,inf,0}
	SetVariable trSetXOffset	,value=_NUM:0	,limits={-inf,inf,0.1}
	SetVariable trScaleXOffset	,value=_NUM:1	,limits={-inf,inf,kMultiIncrStep}
	SetVariable trShiftXOffset	,value=_NUM:0	,limits={-inf,inf,0}
	SetVariable trScaleNorm		,value=_NUM:0	,limits={-inf,inf,1}
	
	SetVariable trOffxDelta		,value=_NUM:0	,limits={-inf,inf,0}
	SetVariable trOffyDelta		,value=_NUM:0	,limits={-inf,inf,0}
	SetVariable vTraceFilter	,value=_STR:""
	
	PopupMenu ColorSelect		,mode=2	,value="*COLORTABLEPOPNONAMES*"
	PopupMenu ProfileSelect		,mode=1	,value=TraceTool_ProfileWaveList()	,proc=TraceTool_ProfileSelect
	
	ModifyControlList ControlNameList("OffsetPanel",";","tr*")			,disable=2				// disable all cursor related controls for now
	ModifyControlList ControlNameList("OffsetPanel",";","Choose*")		,proc=TraceTool_SwitchOffsetMode,mode = 1
	ModifyControlList ControlNameList("OffsetPanel",";","traceCsr*")	,proc=TraceTool_CursorSelector	,mode = 1
	ModifyControlList ControlNameList("OffsetPanel",";","*Button")		,proc=TraceTool_ButtonFunctions
	ModifyControlList ControlNameList("OffsetPanel",";","vaS*") 		,proc=TraceTool_MultiTraceVars	,format="%g"
	ModifyControlList ControlNameList("OffsetPanel",";","trS*") 		,proc=TraceTool_SingleTraceVars	,format="%g"
	ModifyControlList ControlNameList("OffsetPanel",";","*Title")+"MultiTraceBox;SingleTraceBox;"		,frame=0
#if IgorVersion() >= 7.00
	ModifyControlList ControlNameList("OffsetPanel",";","vaS*")+"ColorSelect;ProfileSelect;" 			,focusRing=0
#endif
	Button ReverseButton		,disable=2*(IgorVersion()<7.00)									// resorting does not work in Igor 6
	
	if (CmpStr(IgorInfo(2), "Macintosh") == 0)
		ModifyControlList ControlNameList("OffsetPanel",";","*") ,fsize=10
	endif
	FetchLatestTraceValues()																	// set variable values from top window
	return 0
End
//+++++++++++++++++++++++++++++++++++++ popup help functions +++++++++++++++++++++++++++++++++++++
Function/S TraceTool_ProfileWaveList()
	return "none;"+WaveList("*",";", "DIMS:1")
End

//################################################################################################

static Function/S getTopGraph()
	return StringFromList(0,WinList("*", ";", "WIN:1"))
End

//################################################################################################

static Function getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
	String gName, trace
	Variable &xOff, &yOff, &xScale, &yScale
	String info = TraceInfo(gName,trace,0)
	sscanf StringByKey("offset(x)", info, "="), "{%f,%f}", xOff, yOff
	sscanf StringByKey("muloffset(x)", info, "="), "{%f,%f}", xScale, yScale
	yScale = yScale == 0? 1 : yScale
	xScale = xScale == 0? 1 : xScale
	return 0
End

//################################################################################################

static Function getTraceOffset(gName, trace, axis)
	String gName, trace
	Variable axis
	Variable xOff = 0, yOff = 0
	sscanf StringByKey("offset(x)", TraceInfo(gName,trace,0), "="), "{%f,%f}", xOff, yOff
	return axis ? yOff : xOff
End

//################################ panel update helper function ##################################

Function OffsetPanelUpdateFunc(s)
	STRUCT WMWinHookStruct &s
	Variable HookTakeover = 0
	Switch (s.EventCode)
		case 0:		// activate
			HookTakeover = 1
			FetchLatestTraceValues()
		break
		case 11:	// keys
			String gName = getTopGraph()
			if (!strlen(gName))
				return 0
			endif
			HookTakeover = 1
			String Key = ""
			Switch(s.keycode)
				case 11:	// page up
					Key = "trNextTraceButton"
				break
				case 12:	// page down
					Key = "trPrevTraceButton"
				break
				case 104:	// h
					Key = "trGoToMaxButton"
				break
				case 108:	// l
					Key = "trGoToMinButton"
				break
			EndSwitch
			TraceTool_ExecuteButtonAction(Key)
			
			if (s.keycode > 96 && s.keycode < 101)	// a-d keys
				Key = StringFromList(s.keycode-97, "A;B;C;D;")
				if (strlen(CsrInfo($Key,gName)))
					TraceTool_SwitchCurrentCursor("traceCsr"+Key)
				endif
			endif
			
			if (s.keycode > 27 && s.keycode < 32)	// left, right, up, down
				HookTakeover = TraceTool_KeypadShiftandScale(s.keycode)
			endif
		break
	EndSwitch
	return HookTakeover
End

//################################################################################################

Function TraceTool_KeypadShiftandScale(key)
	Variable key
	
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	
	String trace = GetUserData("OffsetPanel", "", "trace")										// get current trace name and cursor from panel user data
	String csr   = GetUserData("OffsetPanel", "", "cursor")
	if (strlen(trace) == 0 || strlen(csr) == 0)
		return 0
	endif
	Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0
	getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
	Variable yScaleMult = kMultiIncrStep*10^GetValuesOrderOfMagnitude(yScale) 
	Variable xScaleMult = kMultiIncrStep*10^GetValuesOrderOfMagnitude(xScale) 
	
	Wave work  = TraceNameToWaveRef(gName, trace)
	Wave/Z w_x = XWaveRefFromTrace(gName, trace)
	Variable xDelta = DimDelta(work,0)															// delta shift values for cursor control
	Variable yDelta = WaveMax(work)*yScale*10*kOffsetIncrStep
	if (WaveExists(w_x))
		xDelta = abs(leftx(w_x)-rightx(w_x))/(numpnts(w_x))
	endif
	
	Variable neg = 1, mulDelta
	Switch(key)
		case 28:	// left
			neg = -1
		case 29:	// right
			if(GetKeyState(0) & 2^0)
				mulDelta = xScale*neg == -xScaleMult/5 ? xScaleMult*2/5 : xScaleMult/5			// don't scale to exactly zero (which will flip back to 1)
				ModifyGraph muloffset($trace)={xScale+neg*mulDelta,}
			else
				ModifyGraph offset($trace)={xOff+neg*xDelta,}
			endif
		break
		case 31:	// down
			neg = -1
		case 30:	// up
			if(GetKeyState(0) & 2^0)
				mulDelta = yScale*neg == -yScaleMult/5 ? yScaleMult*2/5 : yScaleMult/5
				ModifyGraph muloffset($trace)={,yScale+neg*mulDelta}
			else
				ModifyGraph offset($trace)={,yOff+neg*yDelta}
			endif
		break
	EndSwitch
	FetchLatestCursorValues(csr)
	AdjustProfileAxisScaling()
	return 1
End

//################################################################################################

Function TraceTool_SwitchOffsetMode(s) : CheckBoxControl
	STRUCT WMCheckboxAction &s
	if(s.eventCode == 2)
		CheckBox $StringFromList(0,RemoveFromList(s.ctrlName, ControlNameList("OffsetPanel",";","Choose*"))) ,win=OffsetPanel ,value=0		
	endif
	return 0
End

//################################################################################################

Function TraceTool_CursorSelector(s) : CheckBoxControl
	STRUCT WMCheckboxAction &s
	if(s.eventCode == 2)
		TraceTool_SwitchCurrentCursor(s.ctrlName)
	endif
	return 0
End

//################################################################################################

Function TraceTool_SwitchCurrentCursor(which)
	String which
	Variable i
	String ctrl = RemoveFromList(which, ControlNameList("OffsetPanel",";","traceCsr*"))
	for (i = 0; i < ItemsInList(ctrl); i += 1)
		CheckBox $StringFromList(i,ctrl) ,win=OffsetPanel ,value=0
	endfor
	CheckBox $which ,win=OffsetPanel ,value=1
	FetchLatestCursorValues(ReplaceString("traceCsr", which, ""))
	return 0
End

//################################################################################################

Static Function FetchLatestTraceValues()														// update both left and right side values (if cursors are set)
	String gName = getTopGraph()
	if (!strlen(gName))
		DoWindow /T OffsetPanel, "Modify Traces: No Graph"
		return 0
	endif
	Variable xRange, yRange
	String xUnit = "", yUnit = ""
	Variable error = GetAxisRange(xRange, yRange, xUnit, yUnit, gName)
	if (error)
		return -1
	endif
	xUnit = SelectString(strlen(xUnit),""," "+xUnit)
	yUnit = SelectString(strlen(yUnit),""," "+yUnit)
	
	String traces = ""
	Variable items = FetchTraces(gName,traces,0)
	Variable xOff = 0, yOff = 0, xShift = 0,  yShift = 0
	sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-1,traces),0), "="), "{%f,%f}", xShift, yShift
	if (items > 1)																				// calculate shift and offset from first two traces
		sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-2,traces),0), "="), "{%f,%f}", xOff, yOff
	else
		xOff = xShift; yOff = yShift
	endif
	
	DoWindow /T OffsetPanel, "Modify Traces: "+gName
	SetVariable vaSetYOffset	,win=OffsetPanel	,value=_NUM:yOff-yShift	,format="%g"+yUnit	,limits={-inf,inf,yRange*kOffsetIncrStep}	,userdata=num2str(yOff-yShift)
	SetVariable vaShiftYOffset	,win=OffsetPanel	,value=_NUM:yShift		,format="%g"+yUnit	,limits={-inf,inf,yRange*kOffsetIncrStep}
	SetVariable vaSetXOffset	,win=OffsetPanel	,value=_NUM:xOff-xShift	,format="%g"+xUnit	,limits={-inf,inf,xRange*kOffsetIncrStep}	,userdata=num2str(xOff-xShift)
	SetVariable vaShiftXOffset	,win=OffsetPanel	,value=_NUM:xShift		,format="%g"+xUnit	,limits={-inf,inf,xRange*kOffsetIncrStep*10}
	PopupMenu ProfileSelect 	,win=OffsetPanel 	,mode=1										// update the selected profile or select none
	if (FindListItem("Profileleft", AxisList(gName)) >= 0)
		Wave profile = TraceNameToWaveRef("", "AddTraceProfile")
		PopupMenu ProfileSelect	,win=OffsetPanel	,popmatch=nameofwave(profile)
	endif
	ModifyControlList/Z ControlNameList("OffsetPanel",";","tr*")+"vaApplyAllButton;" ,disable=2	// disable cursor related controls for now
	
	Wave data = TraceNameToWaveRef(gName, StringFromList(0,traces))
	if (!WaveExists(data))																		// probably image data => not useful here
		return 0
	endif
	Button vaApplyAllButton 	,win=OffsetPanel 	,disable=2*(WaveDims(data) != 1)			// this button only works for 1D traces
	//------------------------------------- cursor controls --------------------------------------
	String csr = "A;B;C;D;", currCsr															// fetch available cursor list
	Variable i, selCsr = -1,  setCsr = -1, csrActive = 0
	for (i = 0; i < ItemsInList(csr); i += 1)													// find currently selected cursor
		currCsr = "traceCsr"+StringFromList(i,csr)
		ControlInfo/W=OffsetPanel $currCsr
		if (V_Value)
			CheckBox $currCsr ,win=OffsetPanel ,value=0											// deselect for now
			selCsr = i
			break
		endif
	endfor

	for (i = 0; i < ItemsInList(csr); i += 1)
		currCsr = StringFromList(i,csr)
		if (strlen(CsrInfo($currCsr,gName)))
			if (csrActive == 0 && selCsr == i)													// the currently selected cursor is active and can stay
				csrActive = 1
				setCsr = i
			endif
			if (setCsr == -1)																	// find the first free cursor from left
				setCsr = i
			endif
			CheckBox $("traceCsr"+currCsr) ,win=OffsetPanel ,disable=0
		endif
	endfor
	
	if (setCsr == -1)
		SetWindow OffsetPanel, userdata(cursor)=""												// clear cursor user data
		if (selCsr != -1)
			CheckBox $("traceCsr"+StringFromList(selCsr,csr)) ,win=OffsetPanel ,value=1			// set previous cursor (disabled but keeps cursor selected)
		endif
		return 0
	endif
	//---------------------------------- if cursor was found -------------------------------------
	currCsr = StringFromList(setCsr,csr)
	CheckBox $("traceCsr"+currCsr) ,win=OffsetPanel ,value=1
	String ctrlList = RemoveFromList(ControlNameList("OffsetPanel",";","traceCsr*")+"vaApplyAllButton;",ControlNameList("OffsetPanel",";","tr*"))
	ModifyControlList ctrlList ,disable=0
	FetchLatestCursorValues(currCsr)															// now get cursor specific values (right side)
	return 0
End

//################################################################################################

Static Function FetchLatestCursorValues(csr)
	String csr
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	Variable xRange, yRange, shift
	String xUnit = "", yUnit = "", unit = ""
	Variable error = GetAxisRange(xRange, yRange, xUnit, yUnit, gName)
	if (error)
		return -1
	endif
	
	String trace = StringByKey("TNAME",CsrInfo($csr,gName))
	Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0
	getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
	xUnit = SelectString(strlen(xUnit),""," "+xUnit)
	yUnit = SelectString(strlen(yUnit),""," "+yUnit)
	
	Variable csrShiftx = (hcsr($csr)*xScale+xOff)												// round to prevent floating point errors
	Variable csrShifty = (vcsr($csr)*yScale+yOff)
	csrShiftx = round(csrShiftx*10^6)/10^6
	csrShifty = round(csrShifty*10^6)/10^6
	
	Variable yScOrder = GetValuesOrderOfMagnitude(yScale)
	Variable xScOrder = GetValuesOrderOfMagnitude(xScale) 
	
	Wave data = TraceNameToWaveRef(gName, trace)
	DoWindow /T OffsetPanel, "Modify Traces: " + gName + " - " + ReplaceString("'",trace,"")
	Button trApplyCurButton 	,win=OffsetPanel 	,disable=2*(WaveDims(data) != 1)			// this button only works for 1D traces
	SetVariable trSetYOffset	,win=OffsetPanel	,value=_NUM:yOff				,format="%g"+yUnit	,limits={-inf,inf,yRange*kOffsetIncrStep}
	SetVariable trShiftYOffset	,win=OffsetPanel	,value=_NUM:csrShifty			,format="%g"+yUnit	,limits={-inf,inf,yRange*kOffsetIncrStep*10}
	SetVariable trScaleYOffset	,win=OffsetPanel	,value=_NUM:yScale				,format="%g"		,limits={-inf,inf,kMultiIncrStep*10^yScOrder}
	SetVariable trSetXOffset	,win=OffsetPanel	,value=_NUM:xOff				,format="%g"+xUnit	,limits={-inf,inf,xRange*kOffsetIncrStep}
	SetVariable trShiftXOffset	,win=OffsetPanel	,value=_NUM:csrShiftx			,format="%g"+xUnit	,limits={-inf,inf,xRange*kOffsetIncrStep*10}
	SetVariable trScaleXOffset	,win=OffsetPanel	,value=_NUM:xScale				,format="%g"		,limits={-inf,inf,kMultiIncrStep*10^xScOrder}
	SetVariable trScaleNorm		,win=OffsetPanel	,value=_NUM:vcsr($csr)*yScale	,format="%g"+yUnit
	
	SetVariable trOffxDelta		,win=OffsetPanel	,format="%g"								// first reset unit
	SetVariable trOffyDelta		,win=OffsetPanel	,format="%g"
	shift = TraceTool_FetchXShift(TraceNameToWaveRef(gName, trace),unit)
	if (numtype(shift) == 0)
		SetVariable trOffxDelta	,win=OffsetPanel	,value=_NUM:shift	,format="%g"+SelectString(strlen(unit),""," "+unit)
	endif
	unit = ""
	shift = TraceTool_FetchYShift(TraceNameToWaveRef(gName, trace),unit)
	if (numtype(shift) == 0)
		SetVariable trOffyDelta	,win=OffsetPanel	,value=_NUM:shift	,format="%g"+SelectString(strlen(unit),""," "+unit)
	endif
	
	SetWindow OffsetPanel, userdata(trace)=trace												// save into panel's user data
	SetWindow OffsetPanel, userdata(cursor)=csr
	return 0
End

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Static Function GetAxisRange(xRange, yRange, xUnit, yUnit, gName)
	Variable &xRange, &yRange
	String &xUnit, &yUnit, gName
	String Axes = AxisList(gName), str
	if (WhichListItem("left",Axes) > -1)
		GetAxis/W=$gName/Q left
		yRange = abs(V_max - V_min)
		yUnit  = StringByKey("UNITS", AxisInfo("","left"))
	elseif (WhichListItem("right",Axes) > -1)
		GetAxis/W=$gName/Q right
		yRange = abs(V_max - V_min)
		yUnit  = StringByKey("UNITS", AxisInfo("","right"))
	else
		return -1
	endif
	if (WhichListItem("bottom",Axes) > -1)
		GetAxis/W=$gName/Q bottom
		xRange = abs(V_max - V_min)
		xUnit  = StringByKey("UNITS", AxisInfo("","bottom"))
	elseif (WhichListItem("top",Axes) > -1)
		GetAxis/W=$gName/Q top
		xRange = abs(V_max - V_min)
		xUnit  = StringByKey("UNITS", AxisInfo("","top"))
	else
		return -1
	endif
	sprintf str, "%.*g\r", 1, yRange;	yRange = str2num(str)									// rounding
	sprintf str, "%.*g\r", 1, xRange;	xRange = str2num(str)
	return 0
End

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function GetValuesOrderOfMagnitude(val)
	Variable val
	String NumChar = "", EChar = ""
	SPrintf NumChar, "%.1E", val
	SplitString/E=("[0-9]+.[0-9]([eE][-+]?[0-9]+)?") NumChar, EChar
	Variable OrderOfMagnitude = str2num(EChar[1,strlen(EChar)])
	OrderOfMagnitude = numtype(OrderOfMagnitude) != 0 ? 0 : OrderOfMagnitude
	return OrderOfMagnitude
End

//################################## variable helper function ####################################

Function TraceTool_SingleTraceVars(SV) : SetVariableControl
	STRUCT WMSetVariableAction &SV
	if (SV.eventCode == 1 || SV.eventCode == 2)
		SV.blockReentry=1
		String gName = getTopGraph()
		if (!strlen(gName))
			return 0
		endif
		
		String trace = GetUserData("OffsetPanel", "", "trace")									// get current trace name and cursor from panel user data
		String csr   = GetUserData("OffsetPanel", "", "cursor")
		if (strlen(trace) == 0 || strlen(csr) == 0)
			return 0
		endif
		
		Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0
		getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
		if (StringMatch(SV.ctrlName,"trScale*"))
			SV.dval = abs(SV.dval) < 10e-6 ? sign(SV.dval)*10e-6 : SV.dval						// prevent input from getting very small
		endif
		
		StrSwitch(SV.ctrlName)
			case "trSetYOffset":
				ModifyGraph offset($trace)={,SV.dval}
			break
			case "trSetXOffset":
				ModifyGraph offset($trace)={SV.dval,}
			break
			case "trShiftYOffset":
				ModifyGraph offset($trace)={,(SV.dval-vcsr($csr)*yScale)}
			break
			case "trShiftXOffset":
				ModifyGraph offset($trace)={(SV.dval-hcsr($csr)*xScale),}
			break
			case "trScaleYOffset":
				ModifyGraph muloffset($trace)={,SV.dval}
			break
			case "trScaleXOffset":
				ModifyGraph muloffset($trace)={SV.dval,}
			break
			case "trScaleNorm":
				if (vcsr($csr) != 0)
					ModifyGraph muloffset($trace)={,SV.dval/vcsr($csr)}
				endif
			break
		EndSwitch	
		FetchLatestCursorValues(csr)
		AdjustProfileAxisScaling()
	endif
	return 0
End

//################################################################################################

Function TraceTool_MultiTraceVars(SV) : SetVariableControl
	STRUCT WMSetVariableAction &SV
	if (SV.eventCode == 1 || SV.eventCode == 2)
		SV.blockReentry=1
		String gName = getTopGraph()
		if (!strlen(gName))
			return 0
		endif
		String traces = ""
		Variable items = FetchTraces(gName,traces,0)
		if (items == 0)
			return 0
		endif
		
		Variable PrevVal, curXOff = 0, curYOff = 0, xOff = 0, yOff = 0, YnotX = 0, i
		sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-1,traces),0), "="), "{%f,%f}", xOff, yOff
		StrSwitch(SV.ctrlName)
			case "vaSetYOffset":
				YnotX = 1
			case "vaSetXOffset":
				ControlInfo/W=OffsetPanel ChooseAdd
				Variable addMode = v_Value
				
				PrevVal = str2num(SV.Userdata);	SV.Userdata = num2str(SV.dval)					// get previous offset from and write current value to control user data
				for (i = 0; i < items; i += 1)
					if (addMode)
						sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-1 - i,traces),0), "="), "{%f,%f}", curXOff, curYOff
						if (YnotX)
							ModifyGraph offset($StringFromList(items-1 - i,traces))={,i*(SV.dval-PrevVal)+curYOff}
						else
							ModifyGraph offset($StringFromList(items-1 - i,traces))={i*(SV.dval-PrevVal)+curXOff,}
						endif
					else
						if (YnotX)
							ModifyGraph offset($StringFromList(items-1 - i,traces))={,i*SV.dval+yOff}
						else
							ModifyGraph offset($StringFromList(items-1 - i,traces))={i*SV.dval+xOff,}
						endif
					endif
				endfor
			break
			case "vaScaleYOffset":
				YnotX = 1
			case "vaScaleXOffset":
				PrevVal = str2num(SV.Userdata);	SV.Userdata = num2str(SV.dval)					// get previous multiplier from and write current value to control user data
				PrevVal = PrevVal != 0 ? PrevVal : 1											// make sure multiplier is not zero
				for (i = 0; i < items; i += 1)
					sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-1 - i,traces),0), "="), "{%f,%f}", curXOff, curYOff
					if (YnotX)
						ModifyGraph offset($StringFromList(items-1 - i,traces))={,(curYOff-yOff)*SV.dval/PrevVal+yOff}
					else
						ModifyGraph offset($StringFromList(items-1 - i,traces))={(curXOff-xOff)*SV.dval/PrevVal+xOff,}
					endif
				endfor
			break
			case "vaShiftYOffset":
				YnotX = 1
			case "vaShiftXOffset":
				for (i = 0; i < items; i += 1)
					sscanf StringByKey("offset(x)", TraceInfo(gName,StringFromList(items-1 - i,traces),0), "="), "{%f,%f}", curXOff, curYOff
					if (YnotX)
						ModifyGraph offset($StringFromList(items-1 - i,traces))={,(curYOff-yOff)+SV.dval}
					else
						ModifyGraph offset($StringFromList(items-1 - i,traces))={(curXOff-xOff)+SV.dval,}
					endif
				endfor
			break
		EndSwitch
		String csr = GetUserData("OffsetPanel", "", "cursor")
		if (strlen(csr))
			FetchLatestCursorValues(csr)
		endif
		AdjustProfileAxisScaling()
	endif
	return 0
End

//++++++++++++++++++++++++++++++ help function to get a trace list ++++++++++++++++++++++++++++++

Static Function FetchTraces(win,traces,getAll)
	String win, &traces
	Variable getAll
	
	Variable noHidden = 0, omit = 0
	String filter = ""
	if (WinType("OffsetPanel") && !getAll)														// read settings from panel if open
		ControlInfo/W=OffsetPanel cHiddenTraces
		if (V_flag)
			noHidden = !V_Value
		endif
		ControlInfo/W=OffsetPanel vTraceFilter
		if (V_flag)
			filter = S_Value
		endif
		ControlInfo/W=OffsetPanel cOmitFilter
		if (V_flag)
			omit = V_Value
		endif
	endif
	
	traces = TraceNameList(win,";",1+4*noHidden)
	if (strlen(filter))
		if (omit)
			traces = RemoveFromList(ListMatch(traces,filter),traces)
		else
			traces = ListMatch(traces,filter)
		endif
	endif
	
	Variable i, items = ItemsInList(traces)
	String xAxis, remItem = ""
	for (i = 0; i < items; i += 1)																// remove possible vertical traces
		xAxis = StringByKey("XAXIS",  TraceInfo(win,StringFromList(items-1 - i,traces),0))
		if (!(StringMatch(xAxis,"bottom") || StringMatch(xAxis,"top")))
			remItem += StringFromList(items-1 - i,traces)+";"
		endif
	endfor
	traces = RemoveFromList(remItem,traces)
	traces = RemoveFromListWC(traces, ksTraceIgnoreList)
	return 	ItemsInList(traces)
End

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S RemoveFromListWC(listStr, zapListStr)											// returns listStr, purged of any items that match an item in ZapListStr.
	String listStr, zapListStr
	String removeStr=""
	Variable i
	for(i = 0; i < ItemsInList(zapListStr); i += 1)
		removeStr += ListMatch(listStr, StringFromList(i, zapListStr))							// Wildcards okay! Case insensitive.
	endfor
	return RemoveFromList(removeStr, listStr, ";", 0)
End

//+++++++++++++++++++++++++++++++ help function for Profile Axis +++++++++++++++++++++++++++++++++

Static Function AdjustProfileAxisScaling()
	String gName = getTopGraph()
	if (!strlen(gName) || WhichListItem("Profileleft", AxisList(gName)) < 0)
		return 0
	endif

	String traces = ""
	Variable items = FetchTraces(gName,traces,1)												// get all traces
	if (items == 0)
		return 0
	endif
	Variable currMax = 0, dataMin = 0, i, scale = 0
	for (i = 0; i < items; i += 1)
		Variable yOff = getTraceOffset(gName, StringFromList(i,traces), 1)
		if (yOff > currMax)
			currMax = yOff
			Wave work = TraceNameToWaveRef(gName,StringFromList(i,traces))
			dataMin = WaveMin(work)
		endif
	endfor
	dataMin = numtype(dataMin) != 0 ? 0 : dataMin

	DoUpdate; GetAxis/Q left
	scale = (currMax+dataMin-V_min)/(V_Max-V_min)												// calculate the ratio between highest offset and axis height
	scale = Limit(scale, 0.1, 1)																// prevent scaling to useless values
	ModifyGraph axisEnab(Profileleft)={0,scale}
	return 0
End

//################################### button helper function #####################################

Function TraceTool_ButtonFunctions(SB) : ButtonControl
	STRUCT WMButtonAction &SB
	if (SB.eventCode == 2)
		SB.blockReentry=1
		TraceTool_ExecuteButtonAction(SB.ctrlName)
	endif
	return 0
End

//################################################################################################

Function TraceTool_ExecuteButtonAction(which)
	String which
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	
	String trace = GetUserData("OffsetPanel", "", "trace")										// get current selected trace name and cursor from panel user data
	String csr   = GetUserData("OffsetPanel", "", "cursor")
	Variable xOff = 0, yOff = 0, xScale = 0, yScale = 0, setX, setY
	if (strlen(trace) && strlen(csr))
		getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
	endif
	
	String traces = "",  list = ""
	Variable GoUp = 0, items, i
	StrSwitch(which)
		case "CsrActButton":
			items = FetchTraces(gName,traces,1)													// get all traces
			if (items == 0)
				break
			endif
			
			for (i = 0; i < 4; i += 1)
				String currCsr = StringFromList(i,"A;B;C;D;")
				if (!strlen(CsrInfo($currCsr,gName)))
					trace = StringFromList(0,traces)
					Wave work = TraceNameToWaveRef(gName,trace)
					Cursor/P $currCsr, $trace, DimSize(work,0)/2								// set a new cursor on the center of the first trace
					if (strlen(csr))															// if a cursor was already active, then switch over to the newly placed one
						CheckBox $("traceCsr"+csr)		,win=OffsetPanel ,value=0
						CheckBox $("traceCsr"+currCsr)	,win=OffsetPanel ,value=1
					endif
					FetchLatestTraceValues()
					break
				endif
			endfor
		break
		case "ColorButton":
			ControlInfo/W=OffsetPanel ColorSelect
			String Table = S_value
			ControlInfo/W=OffsetPanel cReverseColor
			ColorSpectrumFromTable((V_Value || GetKeyState(0) & 1), table)
		break
		case "ReverseButton":
			items = FetchTraces(gName,traces,1)													// get all traces
			if (items == 0)
				break
			endif
			for (i = 0; i < items; i += 1)														// reverse trace list
				list += StringFromList(items-1 - i,traces) + ";"
			endfor
			ControlInfo/W=OffsetPanel vaSetYOffset;	setY = V_Value
			ControlInfo/W=OffsetPanel vaSetXOffset;	setX = V_Value
			Make/Free/N=(items)/T w_traces = StringFromList(p,traces)
			Make/Free/N=(items) xOffsets = getTraceOffset("", w_traces[p], 0)-setX*(items-1-2*p)
			Make/Free/N=(items) yOffsets = getTraceOffset("", w_traces[p], 1)-setY*(items-1-2*p)
			String sortTraces = RemoveEnding(ReplaceString(";", list, ","), ",")
			Execute/Z "ReorderTraces _back_,{"+sortTraces+"}"									// execute reverse operation
			if (V_Flag)
				Print "Cannot reverse sorting: Too many traces."
				break
			else
				for (i = 0; i < items; i += 1)													// reapply offsets (in reverse)
					ModifyGraph offset($w_traces[i])={xOffsets[i],yOffsets[i]}
				endfor
				//FetchLatestTraceValues()
			endif
		break
		case "ResetYButton":																	// no break here... reverse button also resets trace offsets
			ModifyGraph offset={,0}//,muloffset={,0}
			SetVariable vaSetYOffset	,win=OffsetPanel	,value=_NUM:0	,userdata="0"
			SetVariable vaScaleYOffset	,win=OffsetPanel	,value=_NUM:1	,userdata="1"
			SetVariable vaShiftYOffset	,win=OffsetPanel	,value=_NUM:0
			AdjustProfileAxisScaling()															// resizes the profile axis
		break
		case "ResetXButton":
			ModifyGraph offset={0,}//,muloffset={0,}
			SetVariable vaSetXOffset	,win=OffsetPanel	,value=_NUM:0	,userdata="0"
			SetVariable vaScaleXOffset	,win=OffsetPanel	,value=_NUM:1	,userdata="1"
			SetVariable vaShiftXOffset	,win=OffsetPanel	,value=_NUM:0
			AdjustProfileAxisScaling()
		break
		case "trResetYButton":
			if (strlen(trace) && strlen(csr))
				ModifyGraph offset($trace)={,0} ,muloffset($trace)={,0}
				SetVariable trSetYOffset	,win=OffsetPanel	,value=_NUM:0
				SetVariable trScaleYOffset	,win=OffsetPanel	,value=_NUM:1
				SetVariable trShiftYOffset	,win=OffsetPanel	,value=_NUM:vcsr($csr)
				SetVariable trScaleNorm		,win=OffsetPanel	,value=_NUM:vcsr($csr)
				AdjustProfileAxisScaling()
			endif
		break
		case "trResetXButton":
			if (strlen(trace) && strlen(csr))
				ModifyGraph offset($trace)={0,} ,muloffset($trace)={0,}
				SetVariable trSetXOffset	,win=OffsetPanel	,value=_NUM:0
				SetVariable trScaleXOffset	,win=OffsetPanel	,value=_NUM:1
				SetVariable trShiftXOffset	,win=OffsetPanel	,value=_NUM:hcsr($csr)
				AdjustProfileAxisScaling()
			endif
		break
		case "trGoToMaxButton":
			GoUp = 1
		case "trGoToMinButton":
			if (strlen(trace) && strlen(csr))
				Wave work = CsrWaveRef($csr,gName)
				WaveStats/Q work
				if (GoUp)
					setX = WaveDims(work) == 2 ? V_maxRowLoc : V_maxLoc
					setY = V_maxColLoc
				else
					setX = WaveDims(work) == 2 ? V_minRowLoc : V_minLoc
					setY = V_minColLoc
				endif
				if (strlen(ImageInfo(gName,trace,0)))
					Cursor/I $csr, $trace, setX, setY											// move cursor to image position
				else
					Cursor	 $csr, $trace, setX													// move cursor to trace position
				endif
				FetchLatestCursorValues(csr)
			endif
		break
		case "trNextTraceButton":
			GoUp = 1
		case "trPrevTraceButton":
			if (strlen(trace) && strlen(csr))
				items = FetchTraces(gName,traces,0)
				setX  = hcsr($csr)
				Variable position = WhichListItem(trace, traces)
				String nextTrace = ""
				if (GoUp)
					if (position < items-1)
						 nextTrace = StringFromList(position+1,traces)
					endif
				else
					if (position > 0)
						 nextTrace = StringFromList(position-1,traces)
					endif
				endif
				if (!strlen(ImageInfo(gName,trace,0)) && strlen(nextTrace))
					Cursor $csr, $nextTrace, setX												// move cursor to next trace
				endif
				FetchLatestCursorValues(csr)
			endif
		break
		case "trAddXButton":
			GoUp = 1
		case "trRemXButton":
			if (strlen(trace) && strlen(csr))
				GoUp = GoUp == 0 ? -1 : GoUp
				ControlInfo/W=OffsetPanel trOffxDelta
				setX = V_Value
				if (setX < 0)
					ModifyGraph muloffset($trace)={-GoUp*abs(xScale),}
				else
					xScale = xScale == -1 ? 0 : xScale
					ModifyGraph muloffset($trace)={abs(xScale),}
				endif
				ModifyGraph offset($trace)={(GoUp*abs(setX)+xOff),}
				FetchLatestCursorValues(csr)
			endif
		break
		case "trAddYButton":
			GoUp = 1
		case "trRemYButton":
			if (strlen(trace) && strlen(csr))
				GoUp = GoUp == 0 ? -1 : GoUp
				ControlInfo/W=OffsetPanel trOffyDelta
				setY = V_Value
				if (setY < 0)
					ModifyGraph muloffset($trace)={,-GoUp*abs(yScale)}
				else
					yScale = yScale == -1 ? 0 : yScale
					ModifyGraph muloffset($trace)={,abs(yScale)}
				endif
				ModifyGraph offset($trace)={,(GoUp*abs(setY)+yOff)}
				AdjustProfileAxisScaling()
				FetchLatestCursorValues(csr)
			endif
		break
		case "trNormTo1Button":
			if (strlen(trace) && strlen(csr))
				if (vcsr($csr) != 0)
					ModifyGraph muloffset($trace)={,1/vcsr($csr)}
					FetchLatestCursorValues(csr)
				endif
			endif
		break
		case "trApplyCurButton":
			if (strlen(trace))																	// falls through with only the current trace selected
				traces = trace
				items  = 1
			endif
		case "vaApplyAllButton":
			if (!strlen(traces))
				items = FetchTraces(gName,traces,0)
			endif
			for (i = 0; i < items; i += 1)
				trace = StringFromList(i,traces)
				getAllTraceOffsets(gName, trace, xOff, yOff, xScale, yScale)
				
				Wave work  = TraceNameToWaveRef(gName, trace)									// apply all scales and offsets to 1D data
				Wave/Z w_x = XWaveRefFromTrace(gName, trace)
				if (WaveDims(work) != 1)														// just in case there is 2D data mixed in
					continue
				endif
				work *= yScale
				work += yOff
				if(WaveExists(w_x))
					w_x *= xScale
					w_x += xOff
				else
					SetScale/P x, DimOffset(work,0) * xScale + xOff, DimDelta(work,0) * xScale, WaveUnits(work,0), work
				endif
				ModifyGraph offset($trace)={0,0} ,muloffset($trace)={0,0}
				
				String lognote = ""																// write a simple log
				lognote += SelectString((xScale!=1),"","\rx-scale="+num2str(xScale))
				lognote += SelectString((xOff!=0),"","\rx-offset="+num2str(xOff))
				lognote += SelectString((yScale!=1),"","\ry-scale="+num2str(yScale))
				lognote += SelectString((yOff!=0),"","\ry-offset="+num2str(yOff))
				if (strlen(lognote))
					lognote[0] = "\rscaling and offsets applied:"
					Note work, lognote
				endif
			endfor
			FetchLatestTraceValues()
		break
	EndSwitch
	return 0
End

//#################################### popup helper function #####################################

Function TraceTool_ProfileSelect(s) : PopupMenuControl
	STRUCT WMPopupAction &s
	if (s.eventCode != 2)
		return 0
	endif
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	SetActiveSubwindow $gName																	// make sure to do stuff in the graph window
	String axes = AxisList(gName)
	if (FindListItem("Profileleft", axes) != -1)												// see if the additional Profile axis is attached
		RemoveFromGraph $("AddTraceProfile")													// remove the old trace (if present)
		if (WhichListItem("bottom",axes) > -1)													// reset the axes and remove the drawn line from previous selections
			ModifyGraph axisEnab(bottom)={0,1}
		endif
		if (WhichListItem("top",axes) > -1)
			ModifyGraph axisEnab(top)={0,1}
		endif
	endif

	StrSwitch(s.popStr)
		case "none":																			// do nothing
			break
		default:
			Wave inwave = $s.popStr
			if (WhichListItem("bottom",axes) > -1)												// make the bottom axis shorter
				ModifyGraph axisEnab(bottom)={0,0.72}
			endif
			if (WhichListItem("top",axes) > -1)
				ModifyGraph axisEnab(top)={0,0.72}
			endif
			AppendToGraph/C=(0,15872,65280)/B=Profilebottom/L=Profileleft/VERT inwave/TN=AddTraceProfile	// append the Profile trace
			ModifyGraph mirror(Profilebottom)=2
			ModifyGraph axisEnab(Profilebottom)={0.76,1}	
			ModifyGraph freePos(Profilebottom)=0 ,freePos(Profileleft)={0.76,kwFraction}		// move the new axes to the right position
			ModifyGraph tkLblRot(Profileleft)=90												// rotate the tick labels
			ModifyGraph minor(Profileleft)=1, tick(Profileleft)=2								// minor ticks
			ModifyGraph standoff=0																// axes are aligned, having no 1 pixel standoff
			ModifyGraph ZisZ(Profilebottom)=1,	ZisZ(Profileleft)=1								// zero is 0 (not 0.00)
			ModifyGraph notation(Profilebottom)=1												// scientific notation (10^4 instead of 10*10^3)
			ModifyGraph tlOffset(Profileleft)=-2												// move the labels in a bit
			AdjustProfileAxisScaling()															// resizes the profile axis
		break
	EndSwitch
	return 0
End

//################################################################################################

Function ColorSpectrumFromTable(rev, table)
	Variable rev
	String table
	
	String traces = ""
	Variable i, items = FetchTraces(getTopGraph(),traces,0)
	if (items == 0)																				// may be an image
		traces = StringFromList(0,ImageNameList("",";"))
		if (strlen(traces) == 0)
			return 0
		endif
		ModifyImage $traces ctab={,,$table,rev}
		return 0
	endif
	
	ColorTab2Wave $table
	Wave M_colors
	for (i = 0; i < items; i += 1)
		Variable row = (i/(items-1))*(DimSize(M_Colors, 0)-1)
		Variable red = M_Colors[row][0], green = M_Colors[row][1], blue = M_Colors[row][2]
		Variable idx = rev == 0 ? i : (items-1)-i
		ModifyGraph/Z rgb($StringFromList(idx,traces)) = (red, green, blue)
	endfor
	KillWaves/Z M_colors
	return 0
End

//________________________________________________________________________________________________
//
//										GRAPH SIZE PANEL
//________________________________________________________________________________________________

Function BuildGraphSizePanel()
	if (WinType("GraphSizePanel"))
		DoWindow/F GraphSizePanel
		return 0
	endif
	
	NewPanel/K=1/W=(500,60,820,415)/N=GraphSizePanel as "Set Graph Size"
	ModifyPanel /W=GraphSizePanel ,fixedSize= 1 ,noEdit=1
	SetWindow GraphSizePanel hook(UpdateHook)=SizePanelUpdateFunc
	
	SetDrawLayer UserBack
	DrawLine 10,230,310,230
	DrawLine 10,290,310,290
	SetDrawEnv linethick= 2,fillpat= 0
	DrawRect 25,10,292,195
	SetDrawEnv linethick= 2,dash= 3,fillpat= 0
	DrawRect 55,40,260,165
	SetDrawEnv linethick= 2,linefgc= (0,0,65535),arrow= 3
	DrawLine 58,70,257,70
	SetDrawEnv linethick= 2,linefgc= (0,0,65535),arrow= 3
	DrawLine 210,43,210,163
	SetDrawEnv linethick= 2,linefgc= (1,39321,19939),arrow= 3
	DrawLine 33,135,285,135
	SetDrawEnv linethick= 2,linefgc= (1,39321,19939),arrow= 3
	DrawLine 110,13,110,193
	
	SetVariable VLeftMargin		,pos={10,95}	,value=_NUM:0
	SetVariable VBottomMargin	,pos={133,170}	,value=_NUM:0
	SetVariable VRightMargin	,pos={250,95}	,value=_NUM:0
	SetVariable VTopMargin		,pos={132,15}	,value=_NUM:0
	SetVariable VGraphHeight	,pos={80,95}	,value=_NUM:0
	SetVariable VPlotWidth		,pos={130,60}	,value=_NUM:0
	SetVariable VGraphWidth		,pos={130,125}	,value=_NUM:0
	SetVariable VPlotHeight		,pos={180,95}	,value=_NUM:0
	SetVariable VMagnification	,pos={220,170}	,value=_NUM:0
	SetVariable VGraphFont		,pos={42,170}	,value=_NUM:0
	
	ModifyControlList ControlNameList("GraphSizePanel",";","*")	,limits={0,2000,1}	,size={60.00,18.00} ,bodyWidth=60 ,proc=Gsize_VarControl
	SetVariable VMagnification	,title="exp:"	,limits={0,8,0} 	,bodyWidth=40	,format="%g x"
	SetVariable VGraphFont		,title="font:"	,limits={0,100,0}	,bodyWidth=40						,help={"Sets the global graph font. If shift+enter is pressed instead the global axes font is changed. Insert 0 to reset to 'auto'"}

	CheckBox cPrint				,pos={35,15}	,size={23,23}	,title="print"				,value=0	,help={"Prints modify commands into history."}
	TitleBox SwapTitle			,pos={145,80}	,size={60,23}	,title="\JCswap:"			,frame=0
	PopupMenu SizeUnit			,pos={195,15}	,size={70,23}	,title="unit: "				,mode=1		,proc=Gsize_PresetPopup		,popvalue="points"	,value= #"\"points;inches;cm\""
	CheckBox SwapAxes			,pos={152,98}	,size={23,23}	,title=""					,value=0	,proc=Gsize_SwapGraph
	
	TitleBox AxisCheckTitle		,pos={15,207}	,size={60,23}	,title="Ticks & Labels:"	,frame=0
	CheckBox cAxis_left			,pos={100,207}	,size={23,23}	,title="Left"				,value=0
	CheckBox cAxis_bottom		,pos={145,207}	,size={23,23}	,title="Bottom"				,value=0
	CheckBox cAxis_right		,pos={210,207}	,size={23,23}	,title="Right"				,value=0
	CheckBox cAxis_top			,pos={265,207}	,size={23,23}	,title="Top"				,value=0
	
	CheckBox KeepAutoMode		,pos={10,267}	,size={23,23}	,title="Keep Auto Mode"		,value=1	,help={"Automatically resets the mode to Auto after changing the size."}
	PopupMenu AspectSelect		,pos={10,237}	,size={145,22}	,title="Aspect Ratio:"	,bodyWidth=70	,proc=Gsize_PresetPopup
	PopupMenu FormatSelect		,pos={135,237}	,size={175,22}	,title="Format:"		,bodyWidth=100	,proc=Gsize_PresetPopup
	Button CopyAllButton		,pos={125,263}	,size={85,22}	,title="Copy Values"
	Button PasteAllButton		,pos={225,263}	,size={85,22}	,title="Paste Copied"

	Button CopyStyleButton		,pos={10,297}	,size={85,22}	,title="Copy Style"						,help={"Copies the style of the top graph."}
	CheckBox CopyAxes			,pos={110,300}	,size={60,22}	,title="w/ Axis Scaling"	,value=1	,help={"Copies the axis scaling as well."}
	PopupMenu GrStyle			,pos={10,327}	,size={200,22}	,title="Style:"			,bodyWidth=165
	Button RevertButton			,pos={225,297}	,size={85,22}	,title="Revert"							,help={"Reverts to last style before Apply was pressed."}
	Button ApplyButton			,pos={225,325}	,size={85,22}	,title="\f01Apply"						,help={"Applies selected style to the top graph."}
	
	PopupMenu GrStyle			,mode=1	,value= Gsize_FetchGraphMacroList()
	PopupMenu AspectSelect		,mode=1	,value= "Free;1:1;-;4:3;3:2;16:9;-;2:3;3:4;4:5"
	PopupMenu FormatSelect		,mode=1	,value= "Free Size;Paper Half;Paper Full;PPT 16:9 Half;PPT 16:9 Full;PPT 4:3 Half;PPT 4:3 Full;"
	
	ModifyControlList ControlNameList("GraphSizePanel",";","*Button")	,proc=Gsize_CopyPaste
	ModifyControlList ControlNameList("GraphSizePanel",";","cAxis_*")	,proc=Gsize_ToggleAxis
	
	if (CmpStr(IgorInfo(2), "Macintosh") == 0)
		ModifyControlList ControlNameList("GraphSizePanel",";","*") ,fsize=10
	endif
	
	NVAR/Z unit = root:$ksGraphSizeUnitSave														// load saved settings
	if (NVAR_Exists(unit))
		PopupMenu SizeUnit ,mode=unit
		String CtrlList = RemoveFromList("VMagnification;VGraphFont;",ControlNameList("GraphSizePanel",";","V*"))
		Variable div  = unit == 1 ? 1 : (unit == 2 ? 72 : 72/2.54)								// in points, inches or cm
		Variable step = unit == 1 ? 1 : (unit == 2 ? 0.014 : 0.036)
		ModifyControlList CtrlList ,limits={0,8000/div,step}									// official max size is 8000 points
	endif

	FetchLatestGraphSizeValues()
	return 0
End
//++++++++++++++++++++++++++++++++ panel update helper function ++++++++++++++++++++++++++++++++++
Function SizePanelUpdateFunc(s)
	STRUCT WMWinHookStruct &s
	If (s.eventCode == 0)
		FetchLatestGraphSizeValues()
		return 1
	endif
	return 0
End

//################################################################################################

Function/S Gsize_FetchGraphMacroList()
	String list = MacroList("*",";","KIND:1,SUBTYPE:GraphStyle")
	String copy = GetUserData("GraphSizePanel", "", "CopyStyle")
	if (strlen(copy))
		list[0] = "Copied Style;"
	endif
	return list
End

//################################################################################################

Function Gsize_SwapGraph(cb) : CheckBoxControl
	STRUCT WMCheckboxAction& cb
	if (cb.eventCode == 2 && strlen(getTopGraph()) > 0)
		ControlInfo/W=GraphSizePanel AspectSelect
		String currAspect = S_Value
		ChangeAspectRatio("Free")																// release the aspect mode and set again later (otherwise swap will switch this too)
		ModifyGraph/Z swapXY=cb.checked
		ControlInfo/W=GraphSizePanel cPrint
		if (V_Value)
			print "•ModifyGraph swapXY="+num2str(cb.checked)
		endif
		ChangeAspectRatio(currAspect)
		FetchLatestGraphSizeValues()
	endif
	return 0
End

//################################################################################################

Function Gsize_ToggleAxis(cb) : CheckBoxControl
	STRUCT WMCheckboxAction& cb
	String gName = getTopGraph()
	if (cb.eventCode == 2 && strlen(gName) > 0)
		SetAxisLabelsAndTicks(ReplaceString("cAxis_",cb.ctrlName,""), gName, !cb.checked)
		FetchLatestGraphSizeValues()
	endif
	return 0
End
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Static Function SetAxisLabelsAndTicks(axName, gName, off)
	String axName, gName
	Variable off
	if (FindListItem(axName,AxisList(gName)) > -1)
		ModifyGraph/W=$gName tick($axName)=3*off
		ModifyGraph/W=$gName noLabel($axName)=1*off
		ModifyGraph/W=$gName margin($axName)=25*off
		ControlInfo/W=GraphSizePanel cPrint
		if (V_Value)
			print "•ModifyGraph tick("+axName+")="+num2str(3*off)+", noLabel("+axName+")="+num2str(1*off)+", margin("+axName+")="+num2str(25*off)
		endif
	endif
	return 0
End

//################################################################################################

Function Gsize_CopyPaste(s) : ButtonControl
	STRUCT WMButtonAction &s
	if (s.eventCode != 2)
		return 0
	endif
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif

	String CtrlList = "", ValList = "", Others = "", AxisSet = "", Style = ""
	Variable i
	StrSwitch(s.ctrlName)
		case "CopyAllButton":
			CtrlList = ControlNameList("GraphSizePanel", ";", "cAxis_*")
			for (i = 0; i < ItemsInList(CtrlList); i += 1)
				ControlInfo/W=GraphSizePanel $StringFromList(i,CtrlList)
				AxisSet += num2str(V_Value) + ";"
			endfor

			CtrlList = ControlNameList("GraphSizePanel", ";", "V*")
			CtrlList = RemoveFromList("VGraphWidth;VGraphHeight;",CtrlList)
			for (i = 0; i < ItemsInList(CtrlList); i += 1)
				ControlInfo/W=GraphSizePanel $StringFromList(i,CtrlList)
				ValList += num2str(V_Value) + ";"
			endfor
			ControlInfo/W=GraphSizePanel SwapAxes
			Others +=  "SwapAxes:" + num2str(V_Value) + ";"
			ControlInfo/W=GraphSizePanel SizeUnit
			Others +=  "SizeUnit:" + S_Value + ";"
			ControlInfo/W=GraphSizePanel AspectSelect
			Others +=  "Aspect:" + S_Value + ";"
			ControlInfo/W=GraphSizePanel FormatSelect
			Others +=  "Format:" + S_Value + ";"
		
			SetWindow GraphSizePanel ,userdata(CtrlSet)=CtrlList								// saved in the panel's user data
			SetWindow GraphSizePanel ,userdata(ValSet)=ValList
			SetWindow GraphSizePanel ,userdata(OtherSet)=Others
			SetWindow GraphSizePanel ,userdata(AxesSet)=AxisSet
		break
		case "PasteAllButton":
			Others   = GetUserData("GraphSizePanel", "", "OtherSet")
			AxisSet	 = GetUserData("GraphSizePanel", "", "AxesSet")
			if (strlen(Others) == 0)
				break
			endif
			Variable swap = str2num(StringByKey("SwapAxes", Others))
			CheckBox SwapAxes  		,win=GraphSizePanel	,value=swap
			PopupMenu SizeUnit 		,win=GraphSizePanel	,popMatch=StringByKey("SizeUnit", Others)
			PopupMenu FormatSelect	,win=GraphSizePanel	,popMatch=StringByKey("Format", Others)
			PopupMenu AspectSelect	,win=GraphSizePanel	,popMatch=StringByKey("Aspect", Others)
			
			String cName
			Variable cValue
			CtrlList = ControlNameList("GraphSizePanel", ";", "cAxis_*")
			for (i = 0; i < ItemsInList(CtrlList); i += 1)
				cName  = ReplaceString("cAxis_",StringFromList(i,CtrlList),"")
				cValue = str2num(StringFromList(i,AxisSet)) == 0
				SetAxisLabelsAndTicks(cName, gName, cValue)
			endfor
			
			CtrlList = GetUserData("GraphSizePanel", "", "CtrlSet")
			ValList	 = GetUserData("GraphSizePanel", "", "ValSet")
			for (i = 0; i < ItemsInList(CtrlList); i += 1)
				cName  = StringFromList(i,CtrlList)
				cValue = str2num(StringFromList(i,ValList))
				SetVariable $cName	,win=GraphSizePanel ,value=_NUM:cValue
				ExecuteVarChange(cName, cValue)
			endfor
			
			ModifyGraph/Z swapXY=swap
			String aspect = StringByKey("Aspect", Others)
			if (!StringMatch(aspect,"Free"))													// actually apply aspect ratio setting as well (should come after swapXY)
				ChangeAspectRatio(aspect)
			endif
		break
		case "ApplyButton":																		// applies selected graph style
			ControlInfo/W=$(s.win) GrStyle
			Style = S_value
			String Last = WinRecreation(gName,0)
			if (StringMatch(Style,"Copied Style"))
				Style = GetUserData(s.win, "", "CopyStyle") 
				for (i = 0; i < ItemsInList(Style); i += 1)
					Execute/Q StringFromList(i,Style)
				endfor
			elseif (strlen(Style))
				Execute/Q Style+"()"
				ControlInfo/W=GraphSizePanel cPrint
				if (V_Value)
					print "•"+Style+"()"
				endif
			endif
			SetWindow $(s.win), userdata(LastgName)=gName										// save into panel's user data
			SetWindow $(s.win), userdata(LastStyle)=Last
		break
		case "RevertButton":																	// copies top-graph's style
			Style = GetUserData(s.win, "", "LastStyle")											// check for existence of backup style
			if (strlen(Style) > 0)
				KillWindow $gName
				String/G ExeCmd = Style
				Execute/P/Q "Execute ExeCmd"
				Execute/P/Q "KillStrings/Z ExeCmd"
			endif
		break
		case "CopyStyleButton":																	// copies top-graph's style
			Style = WinRecreation(gName,1)
			Style = GrepList(Style, "^\t(DefineGuide|ControlBar|SetWindow|Cursor|ListBox|CheckBox|PopupMenu|ValDisplay|SetVariable|Button|NewPanel|ModifyPanel|RenameWindow|SetActiveSubwindow|ShowTools)", 1, "\r") // remove control items
			Style = GrepList(Style, "^\t(ModifyGraph/Z rgb|ModifyGraph/Z muloffset|ModifyGraph/Z offset|ShowInfo)", 1, "\r")
			ControlInfo/W=$(s.win) CopyAxes														// include axes if checkbox is active
			if (!V_Value)
				Style = GrepList(Style, "^\t(SetAxis)", 1, "\r")
			endif
			Style = RemoveListItem(0,Style, "\r")												// remove "proc" line
			Style = RemoveListItem(0,Style, "\r")												// remove "pauseupdate" line
			Style = RemoveListItem(ItemsInList(Style,"\r") - 1,Style, "\r")						// remove "endmacro" line
			Style = ReplaceString("\t", Style, "")
			Style = ReplaceString("\r", Style, ";")
			SetWindow $s.win, userdata(CopyStyle)=Style											// save into panel's user data
			PopupMenu GrStyle ,win=$s.win ,mode= 1
		break
	EndSwitch
	FetchLatestGraphSizeValues()
	return 0
End

//################################################################################################

Function Gsize_VarControl(s) : SetVariableControl
	struct WMSetVariableAction &s
	if (s.eventCode == 1 || s.eventCode == 2)
		ExecuteVarChange(s.ctrlName, s.dval)
	endif
	return 0
End
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Static Function ExecuteVarChange(ctrlName, val)
	String ctrlName
	Variable val
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif

	ControlInfo/W=GraphSizePanel SizeUnit
	Variable modifier = V_Value == 1 ? 1 : (V_Value == 2 ? 72 : 72/2.54)						// in points, inches or cm
	ControlInfo/W=GraphSizePanel KeepAutoMode
	Variable Auto = V_Value																		// reset to Auto scale mode after size change
	ControlInfo/W=GraphSizePanel cPrint
	Variable printcmd = V_Value
	
	String cmd = ""
	StrSwitch(ctrlName)
		case "VLeftMargin":
			ModifyGraph/W=$gName margin(left) = val*modifier
			cmd = "margin(left)="+num2str(val*modifier)
		break
		case "VBottomMargin":
			ModifyGraph/W=$gName margin(bottom) = val*modifier
			cmd = "margin(bottom)="+num2str(val*modifier)
		break
		case "VRightMargin":
			ModifyGraph/W=$gName margin(right) = val*modifier
			cmd = "margin(right)="+num2str(val*modifier)
		break
		case "VTopMargin":
			ModifyGraph/W=$gName margin(top) = val*modifier
			cmd = "margin(top)="+num2str(val*modifier)
		break
		case "VGraphHeight":
			ControlInfo/W=GraphSizePanel VTopMargin
			Variable mTop = V_Value
			ControlInfo/W=GraphSizePanel VBottomMargin
			Variable mBot = V_Value
			Variable pHeight = val-mTop-mBot
			if (val == 0)
				ModifyGraph/W=$gName margin(top) = 0, margin(bottom) = 0,  height = 0
				cmd = "margin(top)=0, margin(bottom)=0,  height=0"
			elseif (pHeight > 0)
				ModifyGraph/W=$gName height = pHeight*modifier
				cmd = "height="+num2str(pHeight*modifier)
			endif
		break
		case "VGraphWidth":
			ControlInfo/W=GraphSizePanel VLeftMargin
			Variable mLeft = V_Value
			ControlInfo/W=GraphSizePanel VRightMargin
			Variable mRight = V_Value
			Variable pWidth = val-mLeft-mRight
			if (val == 0)
				ModifyGraph/W=$gName margin(left) = 0, margin(right) = 0,  width = 0
				cmd = "margin(left)=0, margin(right)=0,  width=0"
			elseif (pWidth > 0)
				ModifyGraph/W=$gName width = pWidth*modifier
				cmd = "width="+num2str(pWidth*modifier)
			endif
		break
		case "VPlotHeight":
			ModifyGraph/W=$gName height = val*modifier
			cmd = "height="+num2str(val*modifier)
		break
		case "VPlotWidth":
			ModifyGraph/W=$gName width = val*modifier
			cmd = "width="+num2str(val*modifier)
		break
		case "VMagnification":
			Variable Mag = val == 1 ? 0 : val	// reset to none
			ModifyGraph/W=$gName expand = Mag
			cmd = "expand="+num2str(Mag)
		break
		case "VGraphFont":
			if ((GetKeyState(0) & 2^2) != 0)	// shift pressed
				ModifyGraph/W=$gName fSize = val
				cmd = "fSize="+num2str(val)
			else
				ModifyGraph/W=$gName gfSize = val
				cmd = "gfSize="+num2str(val)
			endif
		break
	EndSwitch
	
	if (printcmd)
		print "•ModifyGraph "+cmd
	endif
	
	if (Auto)
		DoUpdate/W=$gName
		if (StringMatch(ctrlName,"*Height"))
			ModifyGraph/W=$gName height = 0
		elseif (StringMatch(ctrlName,"*Width"))
			ModifyGraph/W=$gName width = 0
		endif
	endif
	FetchLatestGraphSizeValues()
	return 0
end

//################################################################################################

Function Gsize_PresetPopup(s) : PopupMenuControl
	STRUCT WMPopupAction &s
	if (s.eventCode == 2)
		StrSwitch(s.ctrlName)
			case "AspectSelect":
				ChangeAspectRatio(s.popStr)
			break
			case "FormatSelect":
				ChangeGraphFormat(s.popStr)
			break
			case "SizeUnit":
				String CtrlList = RemoveFromList("VMagnification;VGraphFont;",ControlNameList("GraphSizePanel",";","V*"))
				Variable div  = s.popNum == 1 ? 1 : (s.popNum == 2 ? 72 : 72/2.54)				// in points, inches or cm
				Variable step = s.popNum == 1 ? 1 : (s.popNum == 2 ? 0.014 : 0.036)
				ModifyControlList CtrlList ,win=GraphSizePanel	,limits={0,8000/div,step}		// official max size is 8000 points
				Variable/G root:$ksGraphSizeUnitSave = s.popNum									// save settings into global variable
				FetchLatestGraphSizeValues()
			break
		EndSwitch
	endif
	return 0
End
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Static Function ChangeAspectRatio(which)
	String which
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	String HeightMode = ""
	StrSwitch (which)
		case "Free":
			HeightMode = "0"
		break
		case "1:1":
			HeightMode = "{Aspect,1}"
		break
		case "4:3":
			HeightMode = "{Aspect,3/4}"
		break
		case "3:2":
			HeightMode = "{Aspect,2/3}"
		break
		case "16:9":
			HeightMode = "{Aspect,9/16}"
		break
		case "2:3":
			HeightMode = "{Aspect,3/2}"
		break
		case "3:4":
			HeightMode = "{Aspect,4/3}"
		break
		case "4:5":
			HeightMode = "{Aspect,5/4}"
		break
	EndSwitch
	if (strlen(HeightMode) > 0)
		Execute/Q "ModifyGraph/Z height="+HeightMode+";FetchLatestGraphSizeValues();"
		ControlInfo/W=GraphSizePanel cPrint
		if (V_Value && CmpStr(HeightMode,"0") != 0)
			print "•ModifyGraph height="+HeightMode
		endif
	endif

	return 0
End
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Static Function ChangeGraphFormat(which)
	String which
	String gName = getTopGraph()
	if (!strlen(gName))
		return 0
	endif
	PopupMenu AspectSelect	,win=GraphSizePanel	,popMatch="Free"								// reset aspect popup
	if (StringMatch(which,"Free Size"))
		ModifyGraph/Z width=0, height=0, margin=0
		ModifyGraph/Z gfSize=0, fSize=0, lsize=1
		ModifyGraph/Z expand=0
		FetchLatestGraphSizeValues()
		return 0
	endif
	
	ControlInfo/W=GraphSizePanel KeepAutoMode
	Variable Auto = V_Value
	GetAxis/Q bottom																			// check which axes are active (increases the margin here)
	Variable bottomOn = !V_Flag
	GetAxis/Q left
	Variable leftOn = !V_Flag
	GetAxis/Q right
	Variable rightOn = !V_Flag
	GetAxis/Q top
	Variable topOn = !V_Flag
	
	if (StringMatch(which, "Paper*")) 
		ModifyGraph/Z gfSize=8, fSize=8
		ModifyGraph/Z expand=1.25
		ModifyGraph/Z lsize=1.5
		
		ModifyGraph/Z margin(left)	= (11 + 22*leftOn + 10*(bottomOn || topOn && !leftOn))
		ModifyGraph/Z margin(bottom)= (11 + 22*bottomOn)
		ModifyGraph/Z margin(top)	= (11 + 22*topOn)
		ModifyGraph/Z margin(right)	= (11 + 22*rightOn + 10*(bottomOn || topOn && !rightOn))
		if (StringMatch(which, "*Half"))
			ModifyGraph/Z width=(180 - 12*(leftOn && rightOn))
			ModifyGraph/Z height=(172 - 22*(bottomOn && topOn))
		elseif (StringMatch(which, "*Full"))
			ModifyGraph/Z width=(450 - 12*(leftOn && rightOn))
			ModifyGraph/Z height=(208 - 22*(bottomOn && topOn))
		endif
	endif
	
	if (StringMatch(which, "PPT*")) 
		ModifyGraph/Z gfSize=16, fSize=12
		ModifyGraph/Z expand=0.75
		ModifyGraph/Z lsize=1.5

		ModifyGraph/Z margin(left)	= (18 + 37*leftOn + 12*(bottomOn || topOn && !leftOn))
		ModifyGraph/Z margin(bottom)= (18 + 37*bottomOn)
		ModifyGraph/Z margin(top)	= (18 + 37*topOn)
		ModifyGraph/Z margin(right)	= (18 + 37*rightOn + 12*(bottomOn || topOn && !rightOn))
		if (StringMatch(which, "*16:9*"))
			ModifyGraph/Z height=(360 - 37*(bottomOn && topOn))
			if (StringMatch(which, "*Half"))
				ModifyGraph/Z width=(380 - 25*(leftOn && rightOn))
			elseif (StringMatch(which, "*Full"))
				ModifyGraph/Z width=(800 - 25*(leftOn && rightOn))
			endif
		elseif (StringMatch(which, "*4:3*"))
			ModifyGraph/Z height=(350 - 37*(bottomOn && topOn))
			if (StringMatch(which, "*Half"))
				ModifyGraph/Z width=(290 - 25*(leftOn && rightOn))
			elseif (StringMatch(which, "*Full"))
				ModifyGraph/Z width=(600 - 25*(leftOn && rightOn))
			endif
		endif
	endif

	DoUpdate
	if (Auto)
		ModifyGraph/Z height = 0
		ModifyGraph/Z width = 0
	endif
	FetchLatestGraphSizeValues()
	return 0
End

//################################################################################################

Function FetchLatestGraphSizeValues()
	String gName = getTopGraph(), gRec = ""
	if (!strlen(gName))
		DoWindow /T GraphSizePanel, "Set Graph Size: No Graph"
		return 0
	endif
	
	DoUpdate/W=$gName
	GetWindow $gName gsize																		// values from GetWindow (sometimes off by a fraction?)
	//print V_left, V_right, V_top, V_bottom		// debug
  	Variable gLeft	= V_left
  	Variable gTop	= V_top
  	Variable gWidth	= V_right-V_left	
  	Variable gHeight= V_bottom-V_top	
	GetWindow $gName psize
	//print V_left, V_right, V_top, V_bottom		// debug
	Variable mLeft	= round(V_left - gLeft)
	Variable mTop	= round(V_top - gTop)
	Variable mRight	= round(gWidth - V_right)
	Variable mBottom= round(gHeight - V_bottom)
	Variable pWidth	= round(V_right - V_left)
	Variable pHeight= round(V_bottom - V_top)
	// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	gRec = GrepList(WinRecreation(gName, 0), "^\t(ModifyGraph)", 0 , "\r")						// if available, replace with values from recreation macro (these are exact)
	gRec = GrepList(gRec, "(?i)(height|width|margin|gfsize|swapXY)", 0 , "\r")					// extract only relevant keys
	gRec = ReplaceString("\tModifyGraph ",gRec,"")
	gRec = ReplaceString("\r",gRec,",")

	Variable gSwap = str2num(StringByKey("swapXY", gRec, "=", ","))								// swap axes parameter
	Variable gMagn = abs(str2num(StringByKey("expand", gRec, "=", ",")))						// magnification parameter
	gSwap = numtype(gSwap) == 0 ? gSwap : 0
	gMagn = numtype(gMagn) == 0 ? gMagn : 1
	Variable Rec_mLeft	 = gSwap ? str2num(StringByKey("margin(bottom)", gRec, "=", ",")): str2num(StringByKey("margin(left)", gRec, "=", ","))
	Variable Rec_mTop	 = gSwap ? str2num(StringByKey("margin(right)", gRec, "=", ",")) : str2num(StringByKey("margin(top)", gRec, "=", ","))
	Variable Rec_mRight	 = gSwap ? str2num(StringByKey("margin(top)", gRec, "=", ","))	 : str2num(StringByKey("margin(right)", gRec, "=", ","))
	Variable Rec_mBottom = gSwap ? str2num(StringByKey("margin(left)", gRec, "=", ","))	 : str2num(StringByKey("margin(bottom)", gRec, "=", ","))
	Variable Rec_pWidth	 = gSwap ? str2num(StringByKey("height", gRec, "=", ","))		 : str2num(StringByKey("width", gRec, "=", ","))
	Variable Rec_pHeight = gSwap ? str2num(StringByKey("width", gRec, "=", ","))		 : str2num(StringByKey("height", gRec, "=", ","))
	Variable Rec_gHeight = Rec_pHeight + Rec_mBottom + Rec_mTop
	Variable Rec_gWidth	 = Rec_pWidth  + Rec_mRight  + Rec_mLeft
	Variable Rec_gFont	 = str2num(StringByKey("gfSize", gRec, "=", ","))
	Variable gFont = numtype(Rec_gFont) == 0 ? Rec_gFont : 10
#if IgorVersion() >= 7.00
	GetWindow $gName expand																		// fetch magnification in newer Igor versions
	gMagn = V_Value == 0 ? 1 : V_Value
#endif
	// ++++++++++++++++++++++++++++++++ get aspect ratio info ++++++++++++++++++++++++++++++++++++
	gRec = StringFromList(1, ReplaceString("{", ReplaceString("}",StringByKey(SelectString(gSwap, "height", "width"), gRec, "="),""),""), ",")	// extract aspect ratio
	String PopList = "free;1:1;4:3;3:2;16:9;2:3;3:4;4:5"										// popup match list
	Variable sel = 0																			// extract aspect ratio
	sel += 1 * StringMatch(gRec,"1")
	sel += 2 * StringMatch(gRec,"0.75")
	sel += 3 * StringMatch(gRec,"0.666667")
	sel += 4 * StringMatch(gRec,"0.5625")
	sel += 5 * StringMatch(gRec,"1.5")
	sel += 6 * StringMatch(gRec,"1.33333")
	sel += 7 * StringMatch(gRec,"1.25")
	// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	mLeft	= numtype(Rec_mLeft) == 0 ? Rec_mLeft : mLeft										// if the recreation values are valid then use these
	mBottom	= numtype(Rec_mBottom) == 0 ? Rec_mBottom : mBottom
	mRight	= numtype(Rec_mRight) == 0 ? Rec_mRight : mRight
	mTop	= numtype(Rec_mTop) == 0 ? Rec_mTop : mTop
	pWidth	= numtype(Rec_pWidth) == 0 ? Rec_pWidth : pWidth
	pHeight	= numtype(Rec_pHeight) == 0 ? Rec_pHeight : pHeight
	gHeight	= numtype(Rec_gHeight) == 0 ? Rec_gHeight : pHeight + mTop + mBottom
	gWidth	= numtype(Rec_gWidth) == 0 ? Rec_gWidth : pWidth + mLeft + mRight
	
	ControlInfo/W=GraphSizePanel SizeUnit
	Variable modifier = V_Value == 1 ? 1 : (V_Value == 2 ? 72 : 72/2.54)						// in points, inches or cm
	if (modifier != 1)
		mLeft	= round((mLeft	 /modifier)*1000)/1000
		mBottom	= round((mBottom /modifier)*1000)/1000
		mRight	= round((mRight	 /modifier)*1000)/1000
		mTop	= round((mTop	 /modifier)*1000)/1000
		gHeight	= round((gHeight /modifier)*1000)/1000
		pWidth	= round((pWidth	 /modifier)*1000)/1000
		gWidth	= round((gWidth	 /modifier)*1000)/1000
		pHeight	= round((pHeight /modifier)*1000)/1000
	endif
	// +++++++++++++++++++++++++++++++ apply info to panel controls ++++++++++++++++++++++++++++++
	String undoGName = GetUserData("GraphSizePanel", "", "LastgName")							// check for existence of a undo style for the current graph
	DoWindow /T GraphSizePanel, "Set Graph Size: " + gName
	CheckBox SwapAxes			,win=GraphSizePanel	,value=gSwap
	SetVariable VLeftMargin		,win=GraphSizePanel	,value=_NUM:mLeft	,fstyle=(numtype(Rec_mLeft)==0)
	SetVariable VBottomMargin	,win=GraphSizePanel	,value=_NUM:mBottom	,fstyle=(numtype(Rec_mBottom)==0)
	SetVariable VRightMargin	,win=GraphSizePanel	,value=_NUM:mRight	,fstyle=(numtype(Rec_mRight)==0)
	SetVariable VTopMargin		,win=GraphSizePanel	,value=_NUM:mTop	,fstyle=(numtype(Rec_mTop)==0)
	SetVariable VGraphHeight	,win=GraphSizePanel	,value=_NUM:gHeight	,fstyle=(numtype(Rec_gHeight)==0)
	SetVariable VPlotWidth		,win=GraphSizePanel	,value=_NUM:pWidth	,fstyle=(numtype(Rec_pWidth)==0)
	SetVariable VGraphWidth		,win=GraphSizePanel	,value=_NUM:gWidth	,fstyle=(numtype(Rec_gWidth)==0)
	SetVariable VPlotHeight		,win=GraphSizePanel	,value=_NUM:pHeight	,fstyle=(numtype(Rec_pHeight)==0)
	SetVariable VGraphFont		,win=GraphSizePanel	,value=_NUM:gFont	,fstyle=(numtype(Rec_gFont)==0)
	SetVariable VMagnification	,win=GraphSizePanel	,value=_NUM:gMagn
	PopupMenu AspectSelect		,win=GraphSizePanel	,popMatch=StringFromList(sel,PopList)
	Button RevertButton			,win=GraphSizePanel	,disable=2*(!StringMatch(gName,undoGName))	// disable button if undo backup does not match to current graph
	
	Variable i
	String axes = "left;bottom;right;top;"
	for (i = 0; i < ItemsInList(axes); i += 1)													// set axis checkboxes
		Variable isOn = 0
		String currAxis = StringFromList(i,axes)
		gRec = AxisInfo(gName,currAxis)
		if (strlen(gRec))
			isOn = (str2num(StringByKey("tick(x)",gRec,"=",";"))!=3 && str2num(StringByKey("noLabel(x)",gRec,"=",";"))!= 2)		// inactive if both tick(x)=3 and noLabel(x)=2
		endif
		CheckBox $("cAxis_"+currAxis) ,win=GraphSizePanel ,value=isOn ,disable=2*(strlen(gRec)==0)
	endfor
	return 0
End